#	This file is part of SurrealServices.
#
#	SurrealServices is free software; you can redistribute it and/or modify
#	it under the terms of the GNU General Public License as published by
#	the Free Software Foundation; either version 2 of the License, or
#	(at your option) any later version.
#
#	SurrealServices is distributed in the hope that it will be useful,
#	but WITHOUT ANY WARRANTY; without even the implied warranty of
#	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#	GNU General Public License for more details.
#
#	You should have received a copy of the GNU General Public License
#	along with SurrealServices; if not, write to the Free Software
#	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

package SrSv::MySQL::Stub;

=head1 NAME

SrSv::MySQL::Stub - Create functions for SQL queries

=cut

use strict;

use Symbol 'delete_package';

use SrSv::Debug;
use SrSv::MySQL '$dbh';
use SrSv::Process::Init;

our %types;

sub create_null_stub($) {
	my ($stub) = @_;

	my $sth;
	
	proc_init {
		$sth = $dbh->prepare($stub->{SQL});
	};

	return sub {
		$sth->execute(@_);
	};
}

sub create_scalar_stub($) {
	my ($stub) = @_;

	my $sth;

	proc_init {
		$sth = $dbh->prepare($stub->{SQL});
	};

	return sub {
		$sth->execute(@_);
		return $sth->fetchrow_array;
	};
}

sub create_arrayref_stub($) {
	my ($stub) = @_;

	my $sth;

	proc_init {
		$sth = $dbh->prepare($stub->{SQL});
	};

	return sub {
		$sth->execute(@_);
		return $sth->fetchall_arrayref;
	};
}

BEGIN {
	%types = (
		NULL => \&create_null_stub,
		SCALAR => \&create_scalar_stub,
		ARRAYREF => \&create_arrayref_stub,
	);
}

sub export_stub($$$) {
	my ($name, $proto, $code) = @_;

	no strict 'refs';

	*{$name} = eval "sub $proto { goto &\$code }";
}

sub import {
	my (undef, $ins) = @_;

	while(my ($name, $args) = each %$ins) {
		my $stub = {
			NAME => $name,
			TYPE => $args->[0],
			SQL => $args->[1],
		};
		
		my @params = $stub->{SQL} =~ /\?/g;

		$stub->{PROTO} = '(' . ('$' x @params) . ')';
		print "$stub->{NAME} $stub->{PROTO}\n" if DEBUG;
		
		export_stub scalar(caller) . '::' . $stub->{NAME}, $stub->{PROTO}, $types{$stub->{TYPE}}->($stub);
	}
}

1;

=head1 SYNOPSIS

 use SrSv::MySQL::Stub {
 	get_all_foo => ['ARRAYREF', "SELECT * FROM foo"],
	is_foo_valid => ['SCALAR', "SELECT 1 FROM foo WHERE id=? AND valid=1"],
	delete_foo => ['NULL', "DELETE FROM foo WHERE id=?"],
 };

=head1 DESCRIPTION

This module is a convenient way to make lots of subroutines that execute
SQL statements.

