#	This file is part of SurrealServices.
#
#	SurrealServices is free software; you can redistribute it and/or modify
#	it under the terms of the GNU General Public License as published by
#	the Free Software Foundation; either version 2 of the License, or
#	(at your option) any later version.
#
#	SurrealServices is distributed in the hope that it will be useful,
#	but WITHOUT ANY WARRANTY; without even the implied warranty of
#	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#	GNU General Public License for more details.
#
#	You should have received a copy of the GNU General Public License
#	along with SurrealServices; if not, write to the Free Software
#	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
package misc;
use strict;

sub isint($) {
	my($x) = shift;
	return (int($x) eq $x);
}

sub parse_time($) {
	my ($str) = @_;
	my $out;
	$str =~ s/^\+//;
	$str = lc($str);

	my @vals = split(/(?<!\d)(?=\d+\w)/, $str);

	foreach my $val (@vals) {
		$val =~ /(\d+)(\w)/;
		my ($num, $pos) = ($1, $2);
		
		if($pos eq 'w') { $num *= (86400*7) }
		elsif($pos eq 'd') { $num *= 86400 }
		elsif($pos eq 'h') { $num *= 3600 }
		elsif($pos eq 'm') { $num *= 60 }
		elsif($pos ne 's') { return undef }

		$out += $num;
	}

	return $out;
}

sub split_time($) {
	my ($difference) = @_;
	my ($weeks, $days, $hours, $minutes, $seconds);
	$seconds 	=  $difference % 60;
	$difference 	= ($difference - $seconds) / 60;
	$minutes    	=  $difference % 60;
	$difference 	= ($difference - $minutes) / 60;
	$hours 		=  $difference % 24;
	$difference 	= ($difference - $hours)   / 24;
	$days 		=  $difference % 7;
	$weeks		= ($difference - $days)    /  7;

	return ($weeks, $days, $hours, $minutes, $seconds);
}

sub time_ago($) {
	return time_rel(time() - $_[0]);
}

sub time_rel($) {
	my ($time) = @_;

	my ($weeks, $days, $hours, $minutes, $seconds) = misc::split_time($time);

	if($time >= 604800) {
		return "$weeks week".
			($weeks!=1 ? 's' : '').
			", $days day".
			($days!=1 ? 's' : '');
	}
	elsif($time >= 86400) {
		return "$days day".
			($days!=1 ? 's' : '').
			", $hours hour".
			($hours!=1 ? 's' : '');
	}
	elsif($time >= 3600) {
		return "$hours hour".
			($hours!=1 ? 's' : '').
			", $minutes minute".
			($minutes!=1 ? 's' : '');
	}
	elsif($time >= 60) {
		return "$minutes minute".
		($minutes!=1 ? 's' : '').
		", $seconds second".
		($seconds!=1 ? 's' : '');
	}
	else {
		return "$seconds second".
		($seconds!=1 ? 's' : '');
	}
}

our @months = ( 'Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec' );
our @days = ( 'Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat' );

sub gmtime2(;$) {
	my ($time) = @_;
	$time = time() unless $time;
	my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday) = gmtime($time);
	return $days[$wday].' '.$months[$mon].' '.$mday.' '.($year+1900).' '.
		sprintf("%02d:%02d:%02d GMT", $hour, $min, $sec);
}

sub tz_time($;$) {
	my ($tzoffset, $time) = @_;
	my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday) = gmtime(($time ? $time : time()) + $tzoffset);
	return $days[$wday].' '.$months[$mon].' '.$mday.' '.($year+1900).' '.
		sprintf("%02d:%02d:%02d", $hour, $min, $sec);
}

sub tz_to_offset($) {
	# offset is a signed integer corresponding to 1/4 hr increments
	# making no assumptions of perl optimizing divs to shifts, (x >> 2) is equiv (x / 4)
	return ($_[0] * (3600 >> 2));
}

sub get_date(;$) {
# Gets GMT date, not local date
	my ($time) = @_;
	$time = time() unless $time;
	my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday) = gmtime($time);
	return (!wantarray ? ($year+1900).' '.$months[$mon].' '.$mday : ($year + 1900, $mon+1, $months[$mon], $mday));
}

sub parse_mask($) {
	my ($mask) = @_;
	my ($mnick, $mident, $mhost);

	$mask =~ /^(.*?)(?:\!|\@|$)/;
	$mnick = $1;

	if($mask =~ /\!(.*?)(?:\@|$)/) {
		$mident = $1;
	} else {
		$mident = '';
	}

	if($mask =~ /\@(.*?)$/) {
		$mhost = $1;
	} else {
		$mhost = '';
	}

	return ($mnick, $mident, $mhost);
}

sub parse_hostmask($) {
# This is like parse_mask, but only will parse ident@host
# TKL in particular will use this.
# also could be used to parse email addresses
	my ($mask) = @_;
	my ($mident, $mhost);

	if($mask !~ /@/) {
		return ('', $mask);
	}
	elsif($mask =~ /\!(.*?)(?:\@|$)/) {
		$mident = $1;
	} else {
		$mident = '';
	}

	if($mask =~ /\@(.*?)$/) {
		$mhost = $1;
	} else {
		$mhost = '';
	}

	return ($mident, $mhost);
}

sub glob2sql(@) {
	foreach (@_) {
		s/([%_])/\\$1/g;
		tr/*?/%_/;
	}
	return (wantarray ? @_ : $_[0]);
}

sub sql2glob(@) {
	foreach (@_) {
		s/(?<!\\)_/?/g;
		s/(?<!\\)%/*/g;
		s/\\([%_])/$1/g;
	}
	return (wantarray ? @_ : $_[0]);
}

sub parse_quoted($) {
	my ($in) = @_;
	my @out;

	my @qs = (
		[qr/^\s*\"(.*?)(?<!\\)\"(.*)/,
		  sub { $_[0] =~ s/\\"/\"/g; return $_[0] }],
		[qr/^\s*\/(.*?)(?<!\\)\/(.*)/,
		  sub { $_[0] =~ s#\\/#/#g; return $_[0] }],
		[qr/(\S+)\s*(.*|$)/, undef]
	);

	do {
		foreach my $q (@qs) {
			my $str;
			my ($re, $trans) = @$q;
			
			if(my @x = ($in =~ $re)) {
				($str, $in) = @x;
				$str = &$trans($str) if $trans;
				push @out, $str;
				#print "str: $str\nin: $in\n";
			}
		}
	} while($in =~ /\S/);
	
	return @out;
}

# Some of this may be Unreal/cloak specific, but is mostly generic.
# No IPv6 support yet.
sub make_hostmask($$$$) {
# $type is an integer, 0 - 9
#  0 - *!user@host.domain
#  1 - *!*user@host.domain
#  2 - *!*@host.domain
#  3 - *!*user@*.domain
#  4 - *!*@*.domain
#  5 - nick!user@host.domain
#  6 - nick!*user@host.domain
#  7 - nick!*@host.domain
#  8 - nick!*user@*.domain
#  9 - nick!*@*.domain
	my ($type, $nick, $ident, $host) = @_;

	if($host =~ /^(\d{1,3})\.(\d{1,3})\.(\d{1,3})\.(\d{1,3})$/) {
	# IPv4 address, dotted quad.
		my @octets = ($1, $2, $3, $4);
		if($type =~ /^[3489]$/) {
			$host = $octets[0].'.'.$octets[1].'.*';
		}
	}
	elsif($host =~ /^[A-Z0-9]{7}\.[A-Z0-9]{8}\.[A-Z0-9]{7}\.IP$/) { # should probably be case-sensitive.
	# 74BBBBF2.493EE1E3.CA7BA255.IP
		if($type =~ /^[3489]$/) {
			my @host = split(/\./, $host);
			pop @host; #discard last
			$host = '*.'.$host[2].'.IP'; # Unreal's cloak makes last group be the first two octets.
		}
	} else { 
		# we assume normal hostname
		# We don't know what the cloak prefix will be, nor that it will be sane
		# Or even that we'll have a normal cloaked host (it could be a vhost)
		# So we can't restrict the character-class [much].
		# This could be improved further by popping off the
		# parts that are mostly numbers, if not a normal cloakhost.
		if($type =~ /^[3489]$/) {
			$host =~ /(.+?)\.(.+\.[a-z]{2,3})/i;
			$host = "*.$2";
		}
	}

	if($type =~ /^[1368]$/) {
		$ident =~ s/^\~//;
		$ident = "*$ident";
	} elsif($type =~ /^[2479]$/) {
		$ident = '*';
	}

	if ($type < 5 and $type >= 0) {
		$nick = '*';
	}
	
	return ($nick, $ident, $host);
}

1;
