#	This file is part of SurrealServices.
#
#	SurrealServices is free software; you can redistribute it and/or modify
#	it under the terms of the GNU General Public License as published by
#	the Free Software Foundation; either version 2 of the License, or
#	(at your option) any later version.
#
#	SurrealServices is distributed in the hope that it will be useful,
#	but WITHOUT ANY WARRANTY; without even the implied warranty of
#	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#	GNU General Public License for more details.
#
#	You should have received a copy of the GNU General Public License
#	along with SurrealServices; if not, write to the Free Software
#	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
package thread;

use strict;
no strict 'refs';
use threads;
use threads::shared;
use FreezeThaw qw(freeze thaw);

my $debug = 0;

our $workers : shared = 0;
my @hdlrs;
my $worker;
our $loopback;

our @inqueue : shared;
my @inqueue_ircline : shared;
my @inqueue_wf : shared;

our @outqueue : shared;
our $timercond : shared;

my @working_ircline : shared;
my @working_wf : shared;

sub spawn() {
	module::load();
	my (@workers, $timer, $logger);
	$logger = threads->new(\&log::log_writer);
	log::open_log('diag', 'services.log');
	log::open_log('netdump', 'netdump.log') if main::NETDUMP();

	$working_ircline[0] = $working_wf[0] = undef;
	
	for(my $i; $i < $config::conf{'procs'}; $i++) {
		push @workers, threads->new(\&thread::worker);
		print scalar(@workers), " threads spawned\n";
	}

	$timer = threads->new(\&event::checktimers);

	foreach my $t (@workers) { $t->join; }
	$timer->join;
	$logger->join;

	module::unload_lazy();
	exit();
}

sub wait_for_worker() {
	while(1) {
		lock $workers;
		last if $workers;
		cond_wait($workers);
		last if $workers;
	}
}

sub shutdown() {
	net::agent_quit_all("Shutting down.");
	
	sleep 1;

	$main::status = main::ST_SHUTDOWN();
	
	{
		lock @thread::inqueue;
		cond_broadcast(@thread::inqueue);
	}

	{
		lock @thread::outqueue;
		cond_broadcast(@thread::outqueue);
	}

	{
		lock $thread::timercond;
		cond_broadcast($thread::timercond);
	}
	log::log_shutdown();
}

sub alive() {
	if($main::status == main::ST_PRECONNECT() or $main::status == main::ST_NORMAL()) {
		return 1;
	}
	else {
		return 0;
	}
}

sub ircrecv($) {
	my $line = shift;
	my ($ircline, $line) = split(/ /, $line, 2);
	return unless my @x = net::parse_line($line);
	foreach my $x (@x) {
		callfuncs($x->[0], $x->[1], $x->[2], $x->[4], $ircline, $x->[3]);
	}
}

sub callfuncs($$$$;$$) {
	my ($event, $src, $dst, $args, $ircline, $wf) = @_;
	my @refs = @{$event::hdlr{$event}}; my @args = @$args;
	my $loopback;
	unless($ircline) {
		$loopback = 1;
		$ircline = $net::ircline;
	}
	
	CFOUT: foreach my $r (@refs) {
		foreach my $hdlr (@hdlrs) {
			next CFOUT if($r->[3] eq $hdlr);
		}
		push @hdlrs, $r->[3];
			
		if(
			(!$r->[0] or lc($r->[0]) eq lc($args[$src])) and
			(!$r->[1] or lc($r->[1]) eq lc($args[$dst]) or (
				$r->[1]=~/^#$/i and $args[$dst]=~/^#/i)
			)
		) {
			if($loopback) {
				print "Loopback: ".$r->[3]."\n";
				enter_module(0, $r->[3], @args);
			} else {
				lock @inqueue;
				print "Queue size ".scalar(@inqueue)."\n";
				push @inqueue_ircline, $ircline;
				push @inqueue_wf, $wf;
				push @inqueue, freeze($ircline, $r->[3], @args);
				cond_signal(@inqueue);
			}
		}
		
		pop @hdlrs;
	}
}

sub get_ready_line() {
	my @o;
	
	GRL: for(my $i; $i < @inqueue; $i++) {
		next unless defined($inqueue[$i]);

		my ($ircline, $wf, $line) = ($inqueue_ircline[$i], $inqueue_wf[$i], $inqueue[$i]);
		for(my $j; $j < @working_ircline; $j++) {
			next unless defined($working_ircline[$j]);
			if($working_ircline[$j] < $ircline and $working_wf[$j] != 0 and $working_wf[$j] < $wf) {
				print "Failed to process line $ircline with wf $wf\n";
				next GRL;
			}
		}
		
		@o = ($ircline, $wf, $line);
		undef($inqueue_ircline[$i]);
		undef($inqueue_wf[$i]);
		undef($inqueue[$i]);
		last;
	}

	unless(defined(@o)) {
		print "Can't do anything!\n";
		return undef;
	}

	while(@inqueue and not defined($inqueue[0])) {
		shift @inqueue;
		shift @inqueue_ircline;
		shift @inqueue_wf;
	}

	$working_ircline[threads->self->tid()] = $o[0];
	$working_wf[threads->self->tid()] = $o[1];

	print "Clear to process line ", $o[0], "\n";
	
	return $o[2];
}

sub enter_module(@) {
	my ($ircline, $sub, @args) = @_;
	
	# If this is a loopback, $ircline = 0
	if($ircline) {
		$net::ircline = $ircline;
		$loopback = 0;
	} else {
		$loopback = 1;
	}

	print "enter_module $net::ircline $sub (".join(', ', @args). ")\n";
	
	eval {
		local $SIG{__WARN__} = sub {
			if (main::DEBUG()) {
				net::debug(" -- Warning: ".$_[0], split(/\n/, Carp::longmess($@)), " --");
			} else {
				net::debug(" -- Warning: ".$_[0]);
			}
		};

		local $SIG{__DIE__} = sub {
			net::debug(" --", "-- DIED: ".$_[0], split(/\n/, Carp::longmess($@)), " --");
		};
		
		&{$sub}(@args);
	};

	print "finished with $net::ircline $sub\n";

	net::flushmodes() if($ircline);
}

sub ircsend(@) {
	print "Thread ".threads->self->tid()." generated output.\n" if $debug;
	my $out = join("\r\n", @_);

	{
		lock @outqueue;
		push @outqueue, $out;
		cond_signal(@outqueue);
	}
}

sub worker() {
	$worker = 1;
	module::begin();

	{
		lock $workers;
		$workers++;
		cond_broadcast($workers);
	}
	
	INQUEUE: while(alive()) {
		my $line;
		{
			lock @inqueue;

			$working_ircline[threads->self->tid()] = undef;
			$working_wf[threads->self->tid()] = undef;
			
			while(!($line = get_ready_line())) {
				cond_wait(@inqueue);
				last INQUEUE unless(alive());
			}
		}
		
		print "Thread ".threads->self->tid()." got input.\n" if $debug;
		enter_module(thaw($line));
	}

	{
		lock $workers;
		$workers--;
		cond_broadcast($workers);
	}

	module::end();
	
	print "Worker thread dying!\n";
}

sub writer() {
	OUTQUEUE: while(1) {
		my $line;
		
		while(!defined($line)) {
			lock @outqueue;
	
			unless(@outqueue) {
				last OUTQUEUE unless alive();
				cond_wait(@outqueue);
				last OUTQUEUE unless alive();
			}
	
			$line = shift @outqueue;
		}
	
		net::ircsendreal($line);
	}

	$main::status = main::ST_CLOSED();
}

sub loopback() {
	return $thread::loopback;
}

1;
