#	This file is part of SurrealServices.
#
#	SurrealServices is free software; you can redistribute it and/or modify
#	it under the terms of the GNU General Public License as published by
#	the Free Software Foundation; either version 2 of the License, or
#	(at your option) any later version.
#
#	SurrealServices is distributed in the hope that it will be useful,
#	but WITHOUT ANY WARRANTY; without even the implied warranty of
#	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#	GNU General Public License for more details.
#
#	You should have received a copy of the GNU General Public License
#	along with SurrealServices; if not, write to the Free Software
#	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
package services;
use strict;

use DBI;
use SrSv::MySQL qw($dbh);
use SrSv::Conf qw(main services);
use SrSv::Timer qw(add_timer);
use SrSv::Agent;
use SrSv::IRCd::Event qw(addhandler);

use modules::serviceslibs::adminserv;
use modules::serviceslibs::nickserv;
use modules::serviceslibs::chanserv;
use modules::serviceslibs::operserv;
use modules::serviceslibs::botserv;
use modules::serviceslibs::memoserv;
use modules::serviceslibs::hostserv;

*conf = \%services_conf;

our @agents = (
	['NickServ', '+opqzBHS', 'Nick Registration Agent'],
	['ChanServ', '+pqzBS', 'Channel Registration Agent'],
	['OperServ', '+opqzBHS', 'Operator Services Agent'],
	['MemoServ', '+pqzBS', 'Memo Exchange Agent'],
	['BotServ', '+pqzBS', 'Channel Bot Control Agent'],
	['AdminServ', '+pqzBS', 'Services\' Administration Agent'],
	['HostServ', '+pqzBs', 'vHost Agent']
);
our $qlreason = 'Reserved for Services';

foreach my $a (@agents) {
	agent_connect($a->[0], 'services', undef, $a->[1], $a->[2]);
	net::sqline($a->[0], $qlreason);
	agent_join($a->[0], $main_conf{diag});
	net::setmode($main::rsnick, $main_conf{diag}, '+o', $a->[0]);
}

addhandler('SEOS', undef, undef, 'services::ev_connect');
sub ev_connect {
	botserv::eos();
	nickserv::cleanup();
	nickserv::fix_vhosts();
	chanserv::eos();
	operserv::expire();
}

addhandler('EOS', undef, undef, 'services::eos');
sub eos {
	chanserv::eos($_[0]);
}

addhandler('KILL', undef, undef, 'nickserv::killhandle');

addhandler('NICKCONN', undef, undef, 'services::ev_nickconn');
sub ev_nickconn {
    nickserv::nick_create(@_[0,2..4,8,5,7,9,10,11]);
}

# NickServ
addhandler('NICKCHANGE', undef, undef, 'nickserv::nick_change');
addhandler('QUIT', undef, undef, 'nickserv::nick_delete');
addhandler('UMODE', undef, undef, 'nickserv::umode');
addhandler('CHGHOST', undef, undef, 'nickserv::chghost');
addhandler('CHGIDENT', undef, undef, 'nickserv::chgident');
addhandler('USERIP', undef, undef, 'nickserv::userip');
addhandler('SQUIT', undef, undef, 'nickserv::squit') if $net::noquit;

addhandler('PRIVMSG', undef, 'nickserv', 'nickserv::dispatch');

addhandler('BACK', undef, undef, 'nickserv::notify_auths');

# ChanServ
addhandler('JOIN', undef, undef, 'chanserv::user_join');
addhandler('SJOIN', undef, undef, 'chanserv::handle_sjoin');
addhandler('PART', undef, undef, 'chanserv::user_part');
addhandler('KICK', undef, undef, 'chanserv::process_kick');
addhandler('MODE', undef, qr/^#/, 'chanserv::chan_mode');
addhandler('TOPIC', undef, undef, 'chanserv::chan_topic');

addhandler('PRIVMSG', undef, 'chanserv', 'chanserv::dispatch');

# OperServ
addhandler('PRIVMSG', undef, 'operserv', 'operserv::dispatch');

add_timer('flood_expire', 10, __PACKAGE__, 'operserv::flood_expire');

# MemoServ
addhandler('PRIVMSG', undef, 'memoserv', 'memoserv::dispatch');
addhandler('BACK', undef, undef, 'memoserv::notify');

# BotServ
addhandler('PRIVMSG', undef, undef, 'botserv::dispatch');
addhandler('NOTICE', undef, qr/^#/, 'botserv::chan_msg');

# AdminServ
addhandler('PRIVMSG', undef, 'adminserv', 'adminserv::dispatch');

add_timer('', 30, __PACKAGE__, 'services::maint');
add_timer('', 20, __PACKAGE__, 'nickserv::cleanup_users');
add_timer('', 60, __PACKAGE__, 'nickserv::expire_silence_timed');

# HostServ
addhandler('PRIVMSG', undef, 'hostserv', 'hostserv::dispatch');

# $nick should be a registered root nick, if applicable
# $src is the nick or nickid that sent the command
sub ulog($$$$;$$) {
	my ($service, $level, $text) = splice(@_, 0, 3);
	
	my $hostmask = nickserv::get_hostmask($_[0]);

	# TODO - Record this in the database
	
	log::wlog($service, $level, "$hostmask - $text");
}

sub maint {
	log::wlog($main::rsnick, log::INFO(), " -- Running maintenance routines.");
	add_timer('', 3600, __PACKAGE__, 'services::maint');

	nickserv::expire();
	chanserv::expire();

	log::wlog($main::rsnick, log::INFO(), " -- Maintenance routines complete.");
}

sub init {
	my $dbh = DBI->connect("DBI:mysql:".$services_conf{'mysql-db'}, $services_conf{'mysql-user'}, $services_conf{'mysql-pass'}, {  AutoCommit => 1, RaiseError => 1 });

	$dbh->do("TRUNCATE TABLE chanuser");
	$dbh->do("TRUNCATE TABLE nickchg");
	$dbh->do("TRUNCATE TABLE chan");
	$dbh->do("TRUNCATE TABLE chanban");
	$dbh->do("UPDATE user SET online=0, quittime=".time());

	#$dbh->do("DROP TABLE IF EXISTS olduser");
	#$dbh->do("CREATE TABLE `newuser` (
	#	`id` int(10) unsigned NOT NULL default '0',
	#	`nick` char(30) NOT NULL default '',
	#	`time` int(10) unsigned NOT NULL default '0',
	#	`inval` tinyint(4) NOT NULL default '0',
	#	`ident` char(10) NOT NULL default '',
	#	`host` char(64) NOT NULL default '',
	#	`vhost` char(64) NOT NULL default '',
	#	`server` char(64) NOT NULL default '',
	#	`modes` char(30) NOT NULL default '',
	#	`gecos` char(50) NOT NULL default '',
	#	`guest` tinyint(4) NOT NULL default '0',
	#	PRIMARY KEY  (`id`), UNIQUE KEY `nick` (`nick`)) TYPE=HEAP");
	#$dbh->do("RENAME TABLE user TO olduser, newuser TO user");

	$dbh->disconnect;
}

sub begin {
	nickserv::init();
	chanserv::init();
	operserv::init();
	botserv::init();
	adminserv::init();
	memoserv::init();
	hostserv::init();
}

sub end {
	$dbh->disconnect;
}

sub unload { }

1;
