#	This file is part of SurrealServices.
#
#	SurrealServices is free software; you can redistribute it and/or modify
#	it under the terms of the GNU General Public License as published by
#	the Free Software Foundation; either version 2 of the License, or
#	(at your option) any later version.
#
#	SurrealServices is distributed in the hope that it will be useful,
#	but WITHOUT ANY WARRANTY; without even the implied warranty of
#	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#	GNU General Public License for more details.
#
#	You should have received a copy of the GNU General Public License
#	along with SurrealServices; if not, write to the Free Software
#	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
package chanserv;

use strict;
use DBI qw(:sql_types);

use SrSv::Timer qw(add_timer);

use SrSv::IRCd::State qw($ircline synced initial_synced);
use SrSv::Message qw(message current_message);

use SrSv::Unreal::Modes qw(@opmodes %opmodes);
use SrSv::IRCd::ValidNick qw($valid_nick_re);
use SrSv::Agent;

use SrSv::Shared qw(%enforcers $chanuser_table);

use constant {
	UOP => 1,
	VOP => 2,
	HOP => 3,
	AOP => 4,
	SOP => 5,
	COFOUNDER => 6,
	FOUNDER => 7,

	#current chanreg.flags definition limits us to 16 of these. or 32768 as last flag
	F_OPGUARD => 1,
	F_LEAVEOP => 2,
	F_VERBOSE => 4,
	F_HOLD => 8,
	F_FREEZE => 16,
	F_BOTSTAY => 32,
	F_CLOSE => 64,
	F_DRONE => 128,
	F_SPLITOPS => 256,
	#F_GREET => 512, #deprecated
	#F_NODICE => 1024, #deprecated
	F_LOG => 512,
	F_AUTOVOICE => 1024,
	F_WELCOMEINCHAN => 2048,

	# Maybe this should be a config option
	DEFAULT_BANTYPE => 2,

	CRT_TOPIC => 1,
	CRT_AKICK => 2,
};

*get_user_id = \&nickserv::get_user_id;
*get_user_nick = \&nickserv::get_user_nick;
*is_identified = \&nickserv::is_identified;
*notice = \&nickserv::notice;
*is_online = \&nickserv::is_online;
*get_root_nick = \&nickserv::get_root_nick;

our @levels = ("no", "UOp", "VOp", "HOp", "AOp", "SOp", "co-founder", "founder");
#our @ops = (0, 0, 1, 2, 4, 12, 12, 20);  # normal
our @ops = (0, 0, 1, 2, 4, 8, 16, 16);  # PREFIX_AQ
our @plevels = ('AKICK', 'anyone', 'UOp', 'VOp', 'HOp', 'AOp', 'SOp', 'co-founder', 'founder', 'disabled');
our $plzero = 1;

our @override = (
	['SERVOP',
		{
			ACCCHANGE => 1,
			SET => 1,
			MEMO => 1,
			SETTOPIC => 1,
			AKICK => 1,
			LEVELS => 1
		}
	],
	['SUPER',
		{
			BAN => 1,
			KICK => 1,
			VOICE => 1,
			HALFOP => 1,
			OP => 1,
			ADMIN => 1,
			OWNER => 1,
			SETTOPIC => 1,
			INVITE => 1,
			INVITESELF => 1,
			JOIN => 1,
			CLEAR => 1,
			AKICKENFORCE => 1,
			UPDOWN => 1,
		}
	],
	['HELP',
		{
			ACCLIST => 1,
			LEVELSLIST => 1,
			AKICKLIST => 1,
			INFO => 1,
			GETKEY => 1
		}
	],
	['BOT',
		{
			BOTSAY => 1,
			BOTASSIGN => 1
		}
	]
);

$chanuser_table = 0;

our $csnick = 'ChanServ';
our ($cur_lock, $cnt_lock);
our $err_deny = 'Permission denied.';

#for(my $i=0; $i<@perms; $i++) {
#	my $perm = uc $perms[$i][0];
#	$perms{$perm} = $i;
#}

our (
	$get_joinpart_lock, $get_modelock_lock, $get_update_modes_lock,
	
	$joinpart1, $joinpart2, $chop, $chdeop, $get_op, $get_all_chans,
	$get_all_closed_chans, $get_user_count,

	$is_in_chan,
	
	#$lock_chanuser, $get_all_chan_users,
	$unlock_tables,
	$get_chan_users, $get_chan_users_noacc, $get_chan_users_mask, $get_chan_users_mask_noacc,

	$get_users_nochans, $get_users_nochans_noid,

	$get_using_nick_chans,

	$get_lock, $release_lock, $is_free_lock,

	$chan_create, $chan_delete, $get_chanmodes, $set_chanmodes,

	$is_registered, $get_modelock, $set_modelock, $set_descrip,

	$get_topic, $set_topic, $add_topic,

	$get_acc, $set_acc1, $set_acc2, $del_acc, $get_acc_list, $get_acc_list2, $get_acc_list_mask, $get_acc_list2_mask,
	$wipe_acc_list,
	$get_best_acc, $get_all_acc, $get_highrank, $get_acc_count,

	$get_eos_lock, $get_status_all, $get_status_all_server, $get_modelock_all,

	$get_akick, $get_akick_allchan, $get_akick_alluser, $get_akick_all, $add_akick, $del_akick, $get_akick_list,
	$add_nick_akick, $del_nick_akick, $get_nick_akick, $drop_nick_akick,
	
	$is_level, $get_level, $get_levels, $add_level, $set_level, $reset_level, $clear_levels, $get_level_max,

	$get_founder, $get_successor,
	$set_founder, $set_successor, $del_successor,

	$set_flags, $get_flags, $set_flag, $unset_flag,

	$get_nick_own_chans, $delete_successors,

	$set_lastop, $set_lastused,

	$get_info,

	$register, $drop_acc, $drop_lvl, $drop_akick, $drop,

	$get_expired,

	$get_close, $set_close, $del_close,

	$add_welcome, $del_welcome, $list_welcome, $get_welcomes, $drop_welcome,
	$count_welcome, $consolidate_welcome,

	$add_ban, $delete_bans, $delete_ban, $get_all_bans, $find_bans, $list_bans, $wipe_bans,
	$find_bans_chan_user, $delete_bans_chan_user,

	$add_auth, $list_auth_chan, $get_auth_nick, $get_auth_num, $find_auth,

	$set_bantype, $get_bantype,

	$drop_chantext, $drop_nicktext,
);

sub init() {
	#$chan_create = $services::dbh->prepare("INSERT IGNORE INTO chan SET id=(RAND()*294967293)+1, chan=?");
	$get_joinpart_lock = $services::dbh->prepare("LOCK TABLES chan WRITE, chanuser WRITE");
	$get_modelock_lock = $services::dbh->prepare("LOCK TABLES chanreg READ LOCAL, chan WRITE");
	$get_update_modes_lock = $services::dbh->prepare("LOCK TABLES chan WRITE");
	
	$joinpart1 = $services::dbh->prepare("INSERT IGNORE INTO chanuser SET seq=?, nickid=?, chan=?, joined=1");
	$joinpart2 = $services::dbh->prepare("UPDATE chanuser SET seq=?, joined=?, op=? WHERE nickid=? AND chan=? AND (seq <= ? OR seq > ?)");
	#$chop = $services::dbh->prepare("UPDATE chanuser SET op=op+? WHERE nickid=? AND chan=?");
	$chop = $services::dbh->prepare("UPDATE chanuser SET op=IF(op & ?, op, op ^ ?) WHERE nickid=? AND chan=?");
	$chdeop = $services::dbh->prepare("UPDATE chanuser SET op=IF(op & ?, op ^ ?, op) WHERE nickid=? AND chan=?");
	$get_op = $services::dbh->prepare("SELECT op FROM chanuser WHERE nickid=? AND chan=?");
	$get_all_chans = $services::dbh->prepare("SELECT chan FROM chanuser WHERE nickid=? AND joined=1 AND (seq <= ? OR seq > ?)");
	$get_all_closed_chans = $services::dbh->prepare("SELECT chanclose.chan, chanclose.type, chanclose.reason, chanclose.nick, chanclose.time FROM chanreg, chanuser, chanclose WHERE chanreg.chan=chanuser.chan AND chanreg.chan=chanclose.chan AND chanreg.flags & ? GROUP BY chanclose.chan ORDER BY NULL");
	$get_user_count = $services::dbh->prepare("SELECT COUNT(*) FROM chanuser WHERE chan=? AND joined=1");

	$is_in_chan = $services::dbh->prepare("SELECT 1 FROM chanuser WHERE nickid=? AND chan=? AND joined=1");

	#$lock_chanuser = $services::dbh->prepare("LOCK TABLES chanuser READ, user READ");
	#$get_all_chan_users = $services::dbh->prepare("SELECT user.nick, chanuser.nickid, chanuser.chan FROM chanuser, user WHERE user.id=chanuser.nickid AND chanuser.joined=1");
	$unlock_tables = $services::dbh->prepare("UNLOCK TABLES");

	$get_chan_users = $services::dbh->prepare("SELECT user.nick, user.id FROM chanuser, user
		WHERE chanuser.chan=? AND user.id=chanuser.nickid AND chanuser.joined=1");
	my $chan_users_noacc_tables = 'user '.
		'JOIN chanuser ON (chanuser.nickid=user.id AND chanuser.joined=1 AND user.online=1) '.
		'LEFT JOIN nickid ON (chanuser.nickid=nickid.id) '.
		'LEFT JOIN chanacc ON (nickid.nrid=chanacc.nrid AND chanuser.chan=chanacc.chan)';
	$get_chan_users_noacc = $services::dbh->prepare("SELECT user.nick, user.id FROM $chan_users_noacc_tables
		WHERE chanuser.chan=?
		GROUP BY user.id HAVING MAX(IF(chanacc.level IS NULL, 0, chanacc.level)) <= 0
		ORDER BY NULL");
	my $check_mask = "((user.nick LIKE ?) AND (user.ident LIKE ?)
		AND ((user.vhost LIKE ?) OR (user.host LIKE ?) OR (user.cloakhost LIKE ?)))";
	$get_chan_users_mask = $services::dbh->prepare("SELECT user.nick, user.id FROM chanuser, user
		WHERE chanuser.chan=? AND user.id=chanuser.nickid AND chanuser.joined=1 AND $check_mask");
	$get_chan_users_mask_noacc = $services::dbh->prepare("SELECT user.nick, user.id FROM $chan_users_noacc_tables
		WHERE chanuser.chan=? AND $check_mask
		GROUP BY user.id HAVING MAX(IF(chanacc.level IS NULL, 0, chanacc.level)) <= 0
		ORDER BY NULL");

	$get_users_nochans = $services::dbh->prepare("SELECT user.nick, user.id 
		FROM user LEFT JOIN chanuser ON (chanuser.nickid=user.id AND chanuser.joined=1)
		WHERE chanuser.chan IS NULL AND user.online=1");
	$get_users_nochans_noid = $services::dbh->prepare("SELECT user.nick, user.id
		FROM user LEFT JOIN chanuser ON (chanuser.nickid=user.id AND chanuser.joined=1)
		LEFT JOIN nickid ON (nickid.id=user.id)
		WHERE chanuser.chan IS NULL AND nickid.id IS NULL
		AND user.online=1");

	#$get_using_nick_chans = $services::dbh->prepare("SELECT user.nick FROM user LEFT JOIN nickid ON user.id=nickid.id LEFT JOIN chanuser ON user.id=chanuser.nickid WHERE nickid.nick=? AND chanuser.chan=?");
	$get_using_nick_chans = $services::dbh->prepare("SELECT user.nick FROM user, nickid, nickreg, chanuser
		WHERE user.id=nickid.id AND user.id=chanuser.nickid AND nickid.nrid=nickreg.id AND chanuser.joined=1
		AND nickreg.nick=? AND chanuser.chan=?");

	$get_lock = $services::dbh->prepare("SELECT GET_LOCK(?, 3)");
	$release_lock = $services::dbh->prepare("DO RELEASE_LOCK(?)");
	$is_free_lock = $services::dbh->prepare("SELECT IS_FREE_LOCK(?)");

	$chan_create = $services::dbh->prepare("INSERT IGNORE INTO chan SET seq=?, chan=?");
	$chan_delete = $services::dbh->prepare("DELETE FROM chan WHERE chan=?");
	$get_chanmodes = $services::dbh->prepare("SELECT modes FROM chan WHERE chan=?");
	$set_chanmodes = $services::dbh->prepare("REPLACE INTO chan SET modes=?, chan=?");

	$is_registered = $services::dbh->prepare("SELECT 1 FROM chanreg WHERE chan=?");
	$get_modelock = $services::dbh->prepare("SELECT modelock FROM chanreg WHERE chan=?");
	$set_modelock = $services::dbh->prepare("UPDATE chanreg SET modelock=? WHERE chan=?");

	$set_descrip = $services::dbh->prepare("UPDATE chanreg SET descrip=? WHERE chan=?");

	$get_topic = $services::dbh->prepare("SELECT chantext.data, topicer, topicd FROM chanreg, chantext
		WHERE chanreg.chan=chantext.chan AND chantext.chan=?");
	$set_topic = $services::dbh->prepare("UPDATE chanreg,chantext SET chantext.data=?,
		chanreg.topicer=?, chanreg.topicd=?
		WHERE chanreg.chan=chantext.chan AND chantext.chan=?");
	$add_topic = $services::dbh->prepare("INSERT INTO chantext SET chan=?, type=".CRT_TOPIC().", data=''");

	$get_acc = $services::dbh->prepare("SELECT chanacc.level FROM chanacc, nickalias
		WHERE chanacc.chan=? AND chanacc.nrid=nickalias.nrid AND nickalias.alias=?");
	$set_acc1 = $services::dbh->prepare("INSERT IGNORE INTO chanacc SELECT ?, nrid, ?, NULL, UNIX_TIMESTAMP(), 0
		FROM nickalias WHERE alias=?");
	$set_acc2 = $services::dbh->prepare("UPDATE chanacc, nickalias
		SET chanacc.level=?, chanacc.adder=?, chanacc.time=UNIX_TIMESTAMP()
		WHERE chanacc.chan=? AND chanacc.nrid=nickalias.nrid AND nickalias.alias=?");
	$del_acc = $services::dbh->prepare("DELETE FROM chanacc USING chanacc, nickalias
		WHERE chanacc.chan=? AND chanacc.nrid=nickalias.nrid AND nickalias.alias=?");
	$wipe_acc_list = $services::dbh->prepare("DELETE FROM chanacc WHERE chan=? AND level=?");
	$get_acc_list = $services::dbh->prepare("SELECT nickreg.nick, chanacc.adder, chanacc.time,
		chanacc.last, nickreg.ident, nickreg.vhost
		FROM chanacc, nickreg
		WHERE chanacc.chan=? AND chanacc.level=? AND chanacc.nrid=nickreg.id AND chanacc.level > 0 ORDER BY nickreg.nick");
	$get_acc_list2 = $services::dbh->prepare("SELECT nickreg.nick, chanacc.adder, chanacc.level, chanacc.time,
		chanacc.last, nickreg.ident, nickreg.vhost
		FROM chanacc, nickreg
		WHERE chanacc.chan=? AND chanacc.nrid=nickreg.id AND chanacc.level > 0 ORDER BY nickreg.nick");
	$get_acc_list_mask = $services::dbh->prepare("SELECT IF (nickreg.nick LIKE ?, nickreg.nick, nickalias.alias), chanacc.adder, chanacc.time,
		chanacc.last, nickreg.ident, nickreg.vhost, COUNT(nickreg.id) as c
		FROM chanacc, nickalias, nickreg
		WHERE chanacc.chan=? AND chanacc.level=? AND chanacc.nrid=nickalias.nrid AND nickreg.id=nickalias.nrid
		AND chanacc.level > 0
		AND nickalias.alias LIKE ? AND nickreg.ident LIKE ? AND nickreg.vhost LIKE ?
		GROUP BY nickreg.id
		ORDER BY nickalias.alias");
	$get_acc_list2_mask = $services::dbh->prepare("SELECT IF (nickreg.nick LIKE ?, nickreg.nick, nickalias.alias),
		chanacc.adder, chanacc.level, chanacc.time,
		chanacc.last, nickreg.ident, nickreg.vhost, COUNT(nickreg.id) as c
		FROM chanacc, nickalias, nickreg
		WHERE chanacc.chan=? AND chanacc.nrid=nickalias.nrid AND nickreg.id=nickalias.nrid
		AND chanacc.level > 0
		AND nickalias.alias LIKE ? AND nickreg.ident LIKE ? AND nickreg.vhost LIKE ?
		GROUP BY nickreg.id
		ORDER BY nickalias.alias");

	$get_best_acc = $services::dbh->prepare("SELECT nickreg.nick, chanacc.level
		FROM nickid, nickalias, nickreg, chanacc 
		WHERE nickid.nrid=nickreg.id AND nickalias.nrid=nickreg.id AND nickid.id=?
		AND chanacc.nrid=nickreg.id AND chanacc.chan=? ORDER BY chanacc.level DESC LIMIT 1");
	$get_all_acc = $services::dbh->prepare("SELECT nickreg.nick, chanacc.level
		FROM nickid, nickreg, chanacc
		WHERE nickid.nrid=nickreg.id AND nickid.id=? AND chanacc.nrid=nickreg.id
		AND chanacc.chan=? ORDER BY chanacc.level");
	$get_highrank = $services::dbh->prepare("SELECT user.nick, chanacc.level FROM chanuser, nickid, chanacc, user WHERE chanuser.chan=? AND chanuser.joined=1 AND chanuser.chan=chanacc.chan AND chanuser.nickid=nickid.id AND user.id=nickid.id AND nickid.nrid=chanacc.nrid ORDER BY chanacc.level DESC LIMIT 1");
	$get_acc_count = $services::dbh->prepare("SELECT COUNT(*) FROM chanacc WHERE chan=? AND level=?");

	$get_eos_lock = $services::dbh->prepare("LOCK TABLES akick READ LOCAL, welcome READ LOCAL, chanuser WRITE, user WRITE,
		user AS u1 READ, user AS u2 READ, chan WRITE, chanreg WRITE, nickid READ LOCAL, nickreg READ LOCAL,
		nickalias READ LOCAL, chanacc READ LOCAL, chanban WRITE, svsop READ");
	my $get_status_all_1 = "SELECT chanuser.chan, chanreg.flags, chanreg.bot, user.nick, user.id, user.flags, MAX(chanacc.level), chanuser.op, MAX(nickreg.flags & ".nickserv::F_NEVEROP().")
		FROM chanuser, user, chanreg
		LEFT JOIN nickid ON(nickid.id=chanuser.nickid)
		LEFT JOIN nickreg ON(nickid.nrid=nickreg.id)
		LEFT JOIN chanacc ON(chanacc.chan=chanuser.chan AND chanacc.nrid=nickid.nrid AND (nickreg.flags & ".nickserv::F_NEVEROP().")=0)
		WHERE";
	my $get_status_all_2 = "(user.flags & ".nickserv::UF_FINISHED().")=0 AND chanuser.joined=1 AND (chanreg.flags & ".(F_CLOSE|F_DRONE).") = 0 AND chanreg.chan=chanuser.chan AND user.id=chanuser.nickid AND (nickid.nrid IS NULL OR nickreg.id IS NOT NULL)
		GROUP BY chanuser.chan, chanuser.nickid ORDER BY NULL";
	$get_status_all = $services::dbh->prepare("$get_status_all_1 $get_status_all_2");
	$get_status_all_server = $services::dbh->prepare("$get_status_all_1 user.server=? AND $get_status_all_2");

	$get_modelock_all = $services::dbh->prepare("SELECT chanuser.chan, chan.modes, chanreg.modelock FROM chanreg, chan, chanuser WHERE chanuser.joined=1 AND chanreg.chan=chan.chan AND chanreg.chan=chanuser.chan GROUP BY chanreg.chan ORDER BY NULL");

	my $akick_rows = "user.nick, akick.nick, akick.ident, akick.host, akick.reason";
	my $akick_no_zerolen = "(akick.ident != '' AND akick.host != '')";
	my $akick_single_cond = "$akick_no_zerolen AND user.nick LIKE akick.nick AND user.ident LIKE akick.ident ".
		"AND (user.host LIKE akick.host OR user.vhost LIKE akick.host OR ".
		"(IF(user.cloakhost IS NOT NULL, user.cloakhost LIKE akick.host, 0)))";
	my $akick_multi_cond = "chanuser.chan=akick.chan AND $akick_single_cond";

	$get_akick = $services::dbh->prepare("SELECT $akick_rows FROM akick, user ".
		"WHERE user.id=? AND akick.chan=? AND $akick_single_cond LIMIT 1");
	$get_akick_allchan = $services::dbh->prepare("SELECT $akick_rows FROM $chan_users_noacc_tables
		JOIN akick ON($akick_multi_cond)
		WHERE akick.chan=?
		GROUP BY user.id HAVING MAX(IF(chanacc.level IS NULL, 0, chanacc.level)) <= 0
		ORDER BY NULL");
	$get_akick_alluser = $services::dbh->prepare("SELECT akick.chan, $akick_rows FROM $chan_users_noacc_tables
		JOIN akick ON($akick_multi_cond)
		WHERE chanuser.nickid=?
		GROUP BY user.id HAVING MAX(IF(chanacc.level IS NULL, 0, chanacc.level)) <= 0
		ORDER BY NULL");
	$get_akick_all = $services::dbh->prepare("SELECT akick.chan, $akick_rows FROM $chan_users_noacc_tables
		JOIN akick ON($akick_multi_cond)
		GROUP BY akick.chan, user.id HAVING MAX(IF(chanacc.level IS NULL, 0, chanacc.level)) <= 0
		ORDER BY NULL");
	
	$add_akick = $services::dbh->prepare("INSERT INTO akick SET chan=?, nick=?, ident=?, host=?, adder=?, reason=?, time=UNIX_TIMESTAMP()");
	$add_akick->{PrintError} = 0;
	$del_akick = $services::dbh->prepare("DELETE FROM akick WHERE chan=? AND nick=? AND ident=? AND host=?");
	$get_akick_list = $services::dbh->prepare("SELECT nick, ident, host, adder, reason, time FROM akick WHERE chan=?");

	$add_nick_akick = $services::dbh->prepare("INSERT INTO akick SELECT ?, nickalias.nrid, '', '', ?, ?, UNIX_TIMESTAMP()
		FROM nickalias WHERE alias=?");
	$del_nick_akick = $services::dbh->prepare("DELETE FROM akick USING akick, nickalias
		WHERE akick.chan=? AND akick.nick=nickalias.nrid AND akick.ident='' AND akick.host='' AND nickalias.alias=?");
	$get_nick_akick = $services::dbh->prepare("SELECT reason FROM akick, nickalias
		WHERE akick.chan=? AND akick.nick=nickalias.nrid AND akick.ident='' AND akick.host='' AND nickalias.alias=?");
	$drop_nick_akick = $services::dbh->prepare("DELETE FROM akick USING akick, nickreg
		WHERE akick.nick=nickreg.id AND akick.ident='' AND akick.host='' AND nickreg.nick=?");

	$is_level = $services::dbh->prepare("SELECT 1 FROM chanperm WHERE chanperm.name=?");
	$get_level = $services::dbh->prepare("SELECT IF(chanlvl.level IS NULL, chanperm.level, chanlvl.level), chanlvl.level
		FROM chanperm LEFT JOIN chanlvl ON chanlvl.perm=chanperm.id AND chanlvl.chan=?
		WHERE chanperm.name=?");
	$get_levels = $services::dbh->prepare("SELECT chanperm.name, chanperm.level, chanlvl.level FROM chanperm LEFT JOIN chanlvl ON chanlvl.perm=chanperm.id AND chanlvl.chan=? ORDER BY chanperm.name");
	$add_level = $services::dbh->prepare("INSERT IGNORE INTO chanlvl SELECT ?, chanperm.id, chanperm.level FROM chanperm WHERE chanperm.name=?");
	$set_level = $services::dbh->prepare("UPDATE chanlvl, chanperm SET chanlvl.level=? WHERE chanlvl.chan=? AND chanperm.id=chanlvl.perm AND chanperm.name=?");
	$reset_level = $services::dbh->prepare("DELETE FROM chanlvl USING chanlvl, chanperm WHERE chanperm.name=? AND chanlvl.perm=chanperm.id AND chanlvl.chan=?");
	$clear_levels = $services::dbh->prepare("DELETE FROM chanlvl WHERE chan=?");
	$get_level_max = $services::dbh->prepare("SELECT max FROM chanperm WHERE name=?");

	$get_founder = $services::dbh->prepare("SELECT nickreg.nick FROM chanreg, nickreg WHERE chanreg.chan=? AND chanreg.founderid=nickreg.id");
	$get_successor = $services::dbh->prepare("SELECT nickreg.nick FROM chanreg, nickreg WHERE chanreg.chan=? AND chanreg.successorid=nickreg.id");
	$set_founder = $services::dbh->prepare("UPDATE chanreg, nickreg SET chanreg.founderid=nickreg.id WHERE nickreg.nick=? AND chanreg.chan=?");
	$set_successor = $services::dbh->prepare("UPDATE chanreg, nickreg SET chanreg.successorid=nickreg.id WHERE nickreg.nick=? AND chanreg.chan=?");
	$del_successor = $services::dbh->prepare("UPDATE chanreg SET chanreg.successorid=NULL WHERE chanreg.chan=?");

	$set_flags = $services::dbh->prepare("UPDATE chanreg SET flags=? WHERE chan=?");
	$get_flags = $services::dbh->prepare("SELECT flags FROM chanreg WHERE chan=?");
	$set_flag = $services::dbh->prepare("UPDATE chanreg SET flags=IF(flags & ?, flags, flags ^ ?) WHERE chan=?");
	$unset_flag = $services::dbh->prepare("UPDATE chanreg SET flags=IF(flags & ?, flags ^ ?, flags) WHERE chan=?");

	$get_nick_own_chans = $services::dbh->prepare("SELECT chanreg.chan FROM chanreg, nickreg WHERE nickreg.nick=? AND chanreg.founderid=nickreg.id");
	$delete_successors = $services::dbh->prepare("UPDATE chanreg, nickreg SET chanreg.successorid=NULL WHERE nickreg.nick=? AND chanreg.successorid=nickreg.id");

	$set_lastop = $services::dbh->prepare("UPDATE chanreg SET last=UNIX_TIMESTAMP() WHERE chan=?");
	$set_lastused = $services::dbh->prepare("UPDATE chanacc, nickid SET chanacc.last=UNIX_TIMESTAMP() WHERE 
		chanacc.chan=? AND nickid.id=? AND chanacc.nrid=nickid.nrid AND chanacc.level > 0");

	$get_info = $services::dbh->prepare("SELECT chanreg.descrip, chanreg.regd, chanreg.last, chantext.data, 
		chanreg.topicer, chanreg.modelock, foundernick.nick, successornick.nick, chanreg.bot, chanreg.bantype
		FROM chanreg, nickreg AS foundernick
		LEFT JOIN nickreg AS successornick ON(successornick.id=chanreg.successorid)
		LEFT JOIN chantext ON (chanreg.chan=chantext.chan AND chantext.type=".CRT_TOPIC().")
		WHERE chanreg.chan=? AND foundernick.id=chanreg.founderid");

	$register = $services::dbh->prepare("INSERT INTO chanreg 
		SELECT ?, ?, UNIX_TIMESTAMP(), UNIX_TIMESTAMP(), NULL, NULL,
		NULL, id, NULL, NULL, NULL, ".DEFAULT_BANTYPE()." FROM nickreg WHERE nick=?");
	$register->{PrintError} = 0;
	$drop_acc = $services::dbh->prepare("DELETE FROM chanacc WHERE chan=?");
	$drop_lvl = $services::dbh->prepare("DELETE FROM chanlvl WHERE chan=?");
	$drop_akick = $services::dbh->prepare("DELETE FROM akick WHERE chan=?");
	$drop = $services::dbh->prepare("DELETE FROM chanreg WHERE chan=?");

	$get_expired = $services::dbh->prepare("SELECT chanreg.chan, nickreg.nick FROM chanreg, nickreg LEFT JOIN chanuser ON(chanreg.chan=chanuser.chan AND chanuser.op!=0) WHERE chanreg.founderid=nickreg.id AND chanuser.chan IS NULL AND chanreg.last<? AND !(chanreg.flags & " . F_HOLD . ")");

	$get_close = $services::dbh->prepare("SELECT reason, nick, time FROM chanclose WHERE chan=?");
	$set_close = $services::dbh->prepare("REPLACE INTO chanclose SET chan=?, reason=?, nick=?, time=UNIX_TIMESTAMP(), type=?");
	$del_close = $services::dbh->prepare("DELETE FROM chanclose WHERE chan=?");

	$add_welcome = $services::dbh->prepare("REPLACE INTO welcome SET chan=?, id=?, adder=?, time=UNIX_TIMESTAMP(), msg=?");
	$del_welcome = $services::dbh->prepare("DELETE FROM welcome WHERE chan=? AND id=?");
	$list_welcome = $services::dbh->prepare("SELECT id, time, adder, msg FROM welcome WHERE chan=? ORDER BY id");
	$get_welcomes = $services::dbh->prepare("SELECT msg FROM welcome WHERE chan=? ORDER BY id");
	$drop_welcome = $services::dbh->prepare("DELETE FROM welcome WHERE chan=?");
	$count_welcome = $services::dbh->prepare("SELECT COUNT(*) FROM welcome WHERE chan=?");
	$consolidate_welcome = $services::dbh->prepare("UPDATE welcome SET id=id-1 WHERE chan=? AND id>?");

	$add_ban = $services::dbh->prepare("REPLACE INTO chanban SET chan=?, mask=?, setter=?, type=?, time=UNIX_TIMESTAMP()");
	$delete_bans = $services::dbh->prepare("DELETE FROM chanban WHERE chan=? AND ? LIKE mask AND type=?");
	# likely need a better name for this or for the above.
	$delete_ban = $services::dbh->prepare("DELETE FROM chanban WHERE chan=? AND mask=? AND type=?");
	$find_bans = $services::dbh->prepare("SELECT mask FROM chanban WHERE chan=? AND ? LIKE mask AND type=?");
	$get_all_bans = $services::dbh->prepare("SELECT mask FROM chanban WHERE chan=? AND type=?");
	$list_bans = $services::dbh->prepare("SELECT mask, setter, time FROM chanban WHERE chan=? AND type=?");
	$wipe_bans = $services::dbh->prepare("DELETE FROM chanban WHERE chan=?");

	my $chanban_mask = "((CONCAT(user.nick, '!', user.ident, '\@', user.host) LIKE chanban.mask) ".
			"OR (CONCAT(user.nick , '!' , user.ident , '\@' , user.vhost) LIKE chanban.mask) ".
			"OR IF(user.cloakhost IS NOT NULL, ".
				"(CONCAT(user.nick , '!' , user.ident , '\@' , user.cloakhost) LIKE chanban.mask), 0))";
	$find_bans_chan_user = $services::dbh->prepare("SELECT mask FROM chanban,user
		WHERE chan=? AND user.id=? AND type=? AND $chanban_mask");
	$delete_bans_chan_user = $services::dbh->prepare("DELETE FROM chanban USING chanban,user
		WHERE chan=? AND user.id=? AND type=? AND $chanban_mask");

	$add_auth = $services::dbh->prepare("REPLACE INTO nicktext
		SELECT nickalias.nrid, (".nickserv::NTF_AUTH()."), 1, ?, ? FROM nickalias WHERE nickalias.alias=?");
	$list_auth_chan = $services::dbh->prepare("SELECT nickreg.nick, nicktext.data FROM nickreg, nicktext
		WHERE nickreg.id=nicktext.nrid AND nicktext.type=(".nickserv::NTF_AUTH().") AND nicktext.chan=?");
	$get_auth_nick = $services::dbh->prepare("SELECT nicktext.data FROM nickreg, nickalias, nicktext
		WHERE nickreg.id=nicktext.nrid AND nickreg.id=nickalias.nrid AND nicktext.type=(".nickserv::NTF_AUTH().")
		AND nicktext.chan=? AND nickalias.alias=?");
	$get_auth_num = $services::dbh->prepare("SELECT nickreg.nick, nicktext.data FROM nickreg, nickalias, nicktext
		WHERE nickreg.id=nicktext.nrid AND nickreg.id=nickalias.nrid AND nicktext.type=(".nickserv::NTF_AUTH().")
		AND nicktext.chan=? LIMIT 1 OFFSET ?");
	$get_auth_num->bind_param(2, 0, SQL_INTEGER);
	$find_auth = $services::dbh->prepare("SELECT 1 FROM nickalias, nicktext
		WHERE nickalias.nrid=nicktext.nrid AND nicktext.type=(".nickserv::NTF_AUTH().")
		AND nicktext.chan=? AND nickalias.alias=?");

	$set_bantype = $services::dbh->prepare("UPDATE chanreg SET bantype=? WHERE chan=?");
	$get_bantype = $services::dbh->prepare("SELECT bantype FROM chanreg WHERE chan=?");

	$drop_chantext = $services::dbh->prepare("DELETE FROM chantext WHERE chan=?");
	$drop_nicktext = $services::dbh->prepare("DELETE nicktext.* FROM nicktext WHERE nicktext.chan=?");
}

### CHANSERV COMMANDS ###

sub dispatch($$$) {
	my ($src, $dst, $msg) = @_;
	
	$msg =~ s/^\s+//;
	my @args = split(/\s+/, $msg);
	my $cmd = shift @args;

	my $user = { NICK => $src, AGENT => $csnick };

	return if operserv::flood_check($user);

	if($cmd =~ /^register$/i) {
		if(@args >= 1) {
			my @args = split(/\s+/, $msg, 4);
			cs_register($user, { CHAN => $args[1] }, $args[2], $args[3]);
		} else {
			notice($user, 'Syntax: REGISTER <#channel> [password] [description]');
		}
	}
	elsif($cmd =~ /^(?:[uvhas]op|co?f(ounder)?)$/i) {
		my ($cn, $cmd2) = splice(@args, 0, 2);
		my $chan = { CHAN => $cn };
		
		if($cmd2 =~ /^add$/i) {
			if(@args == 1) {
				cs_xop_add($user, $chan, $cmd, $args[0]);
			} else {
				notice($user, 'Syntax: '.uc $cmd.' <#channel> ADD <nick>');
			}
		}
		elsif($cmd2 =~ /^del(ete)?$/i) {
			if(@args == 1) {
				cs_xop_del($user, $chan, $cmd, $args[0]);
			} else {
				notice($user, 'Syntax: '.uc $cmd.' <#channel> DEL <nick>');
			}
		}
		elsif($cmd2 =~ /^list$/i) {
			if(@args >= 0) {
				cs_xop_list($user, $chan, $cmd, $args[0]);
			} else {
				notice($user, 'Syntax: '.uc $cmd.' <#channel> LIST [mask]');
			}
		}
		elsif($cmd2 =~ /^(wipe|clear)$/i) {
			if(@args == 0) {
				cs_xop_wipe($user, $chan, $cmd);
			} else {
				notice($user, 'Syntax: '.uc $cmd.' <#channel> WIPE');
			}
		}
		else {
			notice($user, 'Syntax: '.uc $cmd.' <#channel> <ADD|DEL|LIST|WIPE>');
		}
	}
	elsif($cmd =~ /^levels$/i) {
		if(@args < 2) {
			notice($user, 'Syntax: LEVELS <#channel> <SET|RESET|LIST|CLEAR>');
			return;
		}

		my $cmd2 = lc(splice(@args, 1, 1));

		if($cmd2 eq 'set') {
			if(@args == 3) {
				cs_levels_set($user, { CHAN => $args[0] }, $args[1], $args[2]);
			} else {
				notice($user, 'Syntax: LEVELS <#channel> SET <permission> <level>');
			}
		}
		elsif($cmd2 eq 'reset') {
			if(@args == 2) {
				cs_levels_set($user, { CHAN => $args[0] }, $args[1]);
			} else {
				notice($user, 'Syntax: LEVELS <#channel> RESET <permission>');
			}
		}
		elsif($cmd2 eq 'list') {
			if(@args == 1) {
				cs_levels_list($user, { CHAN => $args[0] });
			} else {
				notice($user, 'Syntax: LEVELS <#channel> LIST');
			}
		}
		elsif($cmd2 eq 'clear') {
			if(@args == 1) {
				cs_levels_clear($user, { CHAN => $args[0] });
			} else {
				notice($user, 'Syntax: LEVELS <#channel> CLEAR');
			}
		}
		else {
			notice($user, 'Syntax: LEVELS <#channel> <SET|RESET|LIST|CLEAR>');
		}
	}
	elsif($cmd =~ /^akick$/i) {
		if(@args < 2) {
			notice($user, 'Syntax: AKICK <#channel> <ADD|DEL|LIST|WIPE|CLEAR>');
			return;
		}
		
		#my $cmd2 = lc($args[1]);
		my $cmd2 = lc(splice(@args, 1, 1));

		if($cmd2 eq 'add') {
			if(@args >= 2) {
				my @args = split(/\s+/, $msg, 5);
				cs_akick_add($user, { CHAN => $args[1] }, $args[3], $args[4]);
			} else {
				notice($user, 'Syntax: AKICK <#channel> ADD <nick|mask> <reason>');
			}
		}
		elsif($cmd2 eq 'del') {
			if(@args >= 2) {
				cs_akick_del($user, { CHAN => $args[0] }, $args[1]);
			} else {
				notice($user, 'Syntax: AKICK <#channel> DEL <nick|mask>');
			}
		}
		elsif($cmd2 eq 'list') {
			if(@args == 1) {
				cs_akick_list($user, { CHAN => $args[0] });
			} else {
				notice($user, 'Syntax: AKICK <#channel> LIST');
			}
		}
		elsif($cmd2 =~ /^(wipe|clear)$/i) {
			if(@args == 1) {
				cs_akick_wipe($user, { CHAN => $args[0] });
			} else {
				notice($user, 'Syntax: AKICK <#channel> WIPE');
			}
		}
		elsif($cmd2 =~ /^enforce$/i) {
			if(@args == 1) {
				cs_akick_enforce($user, { CHAN => $args[0] });
			} else {
				notice($user, 'Syntax: AKICK <#channel> ENFORCE');
			}
		}
		else {
			notice($user, 'Syntax: AKICK <#channel> <ADD|DEL|LIST|WIPE|CLEAR>');
		}
	}
	elsif($cmd =~ /^info$/i) {
		if(@args == 1) {
			cs_info($user, { CHAN => $args[0] });
		} else {
			notice($user, 'Syntax: INFO <channel>');
		}
	}
	elsif($cmd =~ /^set$/i) {
		if(@args == 2 and lc($args[1]) eq 'unsuccessor') {
			cs_set($user, { CHAN => $args[0] }, $args[1]);
		}
		elsif(@args >= 3 and (
			$args[1] =~ /m(?:ode)?lock/i or
			lc($args[1]) eq 'desc'
		)) {
			my @args = split(/\s+/, $msg, 4);
			cs_set($user, { CHAN => $args[1] }, $args[2], $args[3]);
		}
		elsif(@args == 3) {
			cs_set($user, { CHAN => $args[0] }, $args[1], $args[2]);
		}
		else {
			notice($user, 'Syntax: SET <channel> <option> <value>');
		}
	}
	elsif($cmd =~ /^why$/i) {
		if(@args == 1) {
			cs_why($user, { CHAN => shift @args }, $src);
		}
		elsif(@args >= 2) {
			cs_why($user, { CHAN => shift @args }, @args);
		} else {
			notice($user, 'Syntax: WHY <channel> <nick> [nick [nick ...]]');
			return;
		}
	}
	elsif($cmd =~ /^(de)?(voice|h(alf)?op|op|protect|admin|owner)$/i) {
		if(@args >= 1) {
			cs_setmodes($user, $cmd, { CHAN => shift(@args) }, @args);
		} else {
			notice($user, 'Syntax: '.uc($cmd).' <channel> [nick [nick ...]]');
		}
	}
	elsif($cmd =~ /^(up|down)$/i) {
		cs_updown($user, $cmd, @args);
	}
	elsif($cmd =~ /^drop$/i) {
		if(@args == 1) {
			cs_drop($user, { CHAN => $args[0] });
		} else {
			notice($user, 'Syntax: DROP <channel>');
		}
	}
	elsif($cmd =~ /^help$/i) {
		help::sendhelp($csnick, $src, lc $csnick, @args)
	}
	elsif($cmd =~ /^count$/i) {
		if(@args == 1) {
			cs_count($user, { CHAN => $args[0] });
		} else {
			notice($user, 'Syntax: COUNT <channel>');
		}
	}
	elsif($cmd =~ /^kick$/i) {
		my @args = split(/\s+/, $msg, 4); shift @args;
		if(@args >= 2) {
			cs_kick($user, { CHAN => $args[0] }, $args[1], 0, $args[2])
		}
		else {
			notice($user, 'Syntax: KICK <channel> <nick> [reason]');
		}
	}
	elsif($cmd =~ /^(k(ick)?b(an)?|b(an)?k(ick)?)$/i) {
		my @args = split(/\s+/, $msg, 4); shift @args;
		if(@args >= 2) {
			cs_kick($user, { CHAN => $args[0] }, $args[1], 1, $args[2]);
		} else {
			notice($user, 'Syntax: KICKBAN <channel> <nick> [reason]');
		}
	}
	elsif($cmd =~ /^k(ick)?m(ask)?$/i) {
		my @args = split(/\s+/, $msg, 4); shift @args;
		if(@args >= 2) {
			cs_kickmask($user, { CHAN => $args[0] }, $args[1], 0, $args[2])
		}
		else {
			notice($user, 'Syntax: KICKMASK <channel> <mask> [reason]');
		}
	}
	elsif($cmd =~ /^(k(ick)?b(an)?|b(an)?k(ick)?)m(ask)?$/i) {
		my @args = split(/\s+/, $msg, 4); shift @args;
		if(@args >= 2) {
			cs_kickmask($user, { CHAN => $args[0] }, $args[1], 1, $args[2]);
		} else {
			notice($user, 'Syntax: KICKBANMASK <channel> <mask> [reason]');
		}
	}
	elsif($cmd =~ /^invite$/i) {
		my $chan = shift @args;
		if(@args == 0) {
			cs_invite($user, { CHAN => $chan }, $src)
		}
		elsif(@args >= 1) {
			cs_invite($user, { CHAN => $chan }, @args)
		}
		else {
			notice($user, 'Syntax: INVITE <channel> <nick>');
		}
	}
	elsif($cmd =~ /^(close|forbid)$/i) {
		if(@args > 1) {
			my @args = split(/\s+/, $msg, 3);
			cs_close($user, { CHAN => $args[1] }, $args[2], F_CLOSE);
		}
		else {
			notice($user, 'Syntax: CLOSE <chan> <reason>');
		}
	}
	elsif($cmd =~ /^drone$/i) {
		if(@args > 1) {
			my @args = split(/\s+/, $msg, 3);
			cs_close($user, { CHAN => $args[1] }, $args[2], F_DRONE);
		}
		else {
			notice($user, 'Syntax: DRONE <chan> <reason>');
		}
	}
	elsif($cmd =~ /^clear$/i) {
		my ($cmd, $chan, $clearcmd, $reason) = split(/\s+/, $msg, 4);
		unless ($chan and $clearcmd) {
			notice($user, 'Syntax: CLEAR <channel> <MODES|OPS|USERS|BANS> [reason]');
			return;
		}
		if($clearcmd =~ /^modes$/i) {
			cs_clear_modes($user, { CHAN => $chan }, $reason);
		}
		elsif($clearcmd =~ /^ops$/i) {
			cs_clear_ops($user, { CHAN => $chan }, $reason);
		}
		elsif($clearcmd =~ /^users$/i) {
			cs_clear_users($user, { CHAN => $chan }, $reason);
		}
		elsif($clearcmd =~ /^bans?$/i) {
			cs_clear_bans($user, { CHAN => $chan }, 0, $reason);
		}
		elsif($clearcmd =~ /^excepts?$/i) {
			cs_clear_bans($user, { CHAN => $chan }, 128, $reason);
		}
		else {
			notice($user, "Unknown CLEAR command \002$clearcmd\002", 
				'Syntax: CLEAR <channel> <MODES|OPS|USERS|BANS> [reason]');
		}
	}
	elsif($cmd =~ /^mkick$/i) {
		my ($cmd, $chan, $reason) = split(/\s+/, $msg, 3);
		if($chan) {
			cs_clear_users($user, { CHAN => $chan }, $reason);
		}
		else {
			notice($user, 'Syntax: MKICK <chan> [reason]');
		}
	}
	elsif($cmd =~ /^mdeop$/i) {
		my ($cmd, $chan, $reason) = split(/\s+/, $msg, 3);
		if($chan) {
			cs_clear_ops($user, { CHAN => $chan }, $reason);
		}
		else {
			notice($user, 'Syntax: MDEOP <chan> [reason]');
		}
	}
	elsif($cmd =~ /^welcome$/i) {
		my $wcmd = splice(@args, 1, 1);
		if(lc($wcmd) eq 'add') {
			my ($chan, $wmsg) = (splice(@args, 0, 1), join(' ', @args));
			unless ($chan and $wmsg) {
				notice($user, 'Syntax: WELCOME <channel> ADD <message>');
				return;
			}
			cs_welcome_add($user, { CHAN => $chan }, $wmsg);
		}
		elsif(lc($wcmd) eq 'del') {
			if (@args != 2 or !misc::isint($args[1])) {
				notice($user, 'Syntax: WELCOME <channnel> DEL <number>');
				return;
			}
			cs_welcome_del($user, { CHAN => $args[0] }, $args[1]);
		}
		elsif(lc($wcmd) eq 'list') {
			if (@args != 1) {
				notice($user, 'Syntax: WELCOME <channel> LIST');
				return;
			}
			cs_welcome_list($user, { CHAN => $args[0] });
		}
		else {
			notice($user, 'Syntax: WELCOME <channel> <ADD|DEL|LIST>');
		}
	}
	elsif($cmd =~ /^alist$/i) {
		if(@args >= 1) {
			cs_alist($user, { CHAN => shift @args }, shift @args);
		} else {
			notice($user, 'Syntax: ALIST <channel> [mask]');
		}
	}
	elsif($cmd =~ /^unban$/i) {
		if(@args == 1) {
			cs_unban($user, { CHAN => $args[0] }, $src);
		}
		elsif(@args == 2) {
			cs_unban($user, { CHAN => $args[0] }, $args[1]);
		} else {
			notice($user, 'Syntax: UNBAN <channel> [nick]');
		}
	}
	elsif($cmd =~ /^getkey$/i) {
		if(@args == 1) {
			cs_getkey($user, { CHAN => $args[0] });
		} else {
			notice($user, 'Syntax: GETKEY <channel>');
		}
	}
	elsif($cmd =~ /^auth$/i) {
		if (@args == 0) {
			notice($user, 'Syntax: AUTH <channel> <LIST|DELETE> [param]');
		} else {
			cs_auth($user, { CHAN => shift @args }, shift @args, @args);
		}
	}
	elsif($cmd =~ /^dice$/i) {
		if (@args) {
			my $ret;
			notice($user, $args[0].': '.(($ret = botserv::get_dice(split('d', $args[0]))) > 0 ? $ret : 'Error') );
		} else {
			notice($user, 'Syntax: DICE <num>d<sides>');
		}
	}
	elsif($cmd =~ /^ban$/i) {
		my $chan = shift @args;
		if(@args >= 1) {
			cs_ban($user, { CHAN => $chan }, @args)
		}
		else {
			notice($user, 'Syntax: BAN <channel> <nick|mask>');
		}
	}
	elsif($cmd =~ /^assign$/i) {
		my $chan = shift @args;
		notice($user, "$csnick ASSIGN is deprecated. Please use $botserv::bsnick ASSIGN");
		if(@args == 2) {
			botserv::bs_assign($user, { CHAN => shift @args }, shift @args);
		}
		else {
			notice($user, 'Syntax: ASSIGN <#channel> <bot>');
		}
	}
	else {
		notice($user, "Unrecognized command \002$cmd\002.", "For help, type: \002/msg chanserv help\002");
		log::wlog($csnick, log::DEBUG(), "$src tried to use $csnick $msg");
	}
}

sub cs_register($$;$$) {
	my ($user, $chan, $pass, $desc) = @_;
	# $pass is still passed in, but never used!
	my $src = get_user_nick($user);
	my $cn = $chan->{CHAN};

	unless(is_identified($user, $src)) {
		notice($user, 'You must register your nickname first.', "Type \002/msg NickServ HELP\002 for information on registering nicknames.");
		return;
	}

	unless(is_in_chan($user, $chan)) {
	        notice($user, "You are not in \002$cn\002.");
	        return;
	}

	unless(get_op($user, $chan) & ($opmodes{o} | $opmodes{a} | $opmodes{q})) {
	# This would be preferred to be a 'opmode_mask' or something
	# However that might be misleading due to hop not being enough to register
	        notice($user, "You must have channel operator status to register \002$cn\002.");
		return;
	}

	my $root = get_root_nick($src);

	if($desc) {
		my $dlength = length($desc);
		if($dlength >= 350) {
			notice($user, 'Channel description is too long by '. $dlength-350 .' character(s). Maximum length is 350 characters.');
			return;
		}
	}

	if($register->execute($cn, $desc, $root)) {
		notice($user, ' ',"\002Your channel is now registered. Thank you.\002");
		notice($user, ' ', "\002NOTICE:\002 Channel passwords are not used, as a security precaution.")
			if $pass;
		set_acc($root, $user, $chan, FOUNDER);
		$set_modelock->execute('+rnt', $cn);
		$add_topic->execute($cn);
		do_modelock($chan);
		services::ulog($csnick, log::INFO(), "registered $cn", $user, $chan);
	} else {
		notice($user, 'That channel has already been registered.');
	}
}

sub cs_xop_ad_pre($$$$$) {
	my ($user, $chan, $nick, $level, $del) = @_;
	
	my $old = get_acc($nick, $chan); $old = 0 unless $old;
	my $slevel = get_best_acc($user, $chan);
	
	unless(($del and is_identified($user, $nick)) or adminserv::can_do($user, 'SERVOP')) {
		unless($level < $slevel and $old < $slevel) {
			notice($user, $err_deny);
			return undef;
		}
		can_do($chan, 'ACCCHANGE', undef, $user) or return undef;
	}

	nickserv::chk_registered($user, $nick) or return undef;
	if (nickserv::chk_flag($nick, nickserv::F_NOACC()) and !adminserv::can_do($user, 'SERVOP') and !$del) {
		notice($user, "\002$nick\002 is not able to be added to access lists.");
		return undef;
	}

	return $old;
}

sub cs_xop_list($$$;$) {
	my ($user, $chan, $cmd, $mask) = @_;
	chk_registered($user, $chan) or return;
	my $cn = $chan->{CHAN};
	my $level = xop_byname($cmd);
	
	can_do($chan, 'ACCLIST', undef, $user) or return;

	my @reply;
	if($mask) {
		my ($mnick, $mident, $mhost) = misc::glob2sql(misc::parse_mask($mask));
		$mnick = '%' if($mnick eq '');
		$mident = '%' if($mident eq '');
		$mhost = '%' if($mhost eq '');
		
		$get_acc_list_mask->execute($mnick, $cn, $level, $mnick, $mident, $mhost);
		while(my ($n, $a, $t, $lu, $id, $vh) = $get_acc_list_mask->fetchrow_array) {
			push @reply, "*) $n ($id\@$vh)" . ($a ? ' Added by: '.$a : '');
			push @reply, '      '.($t ? 'Date/time added: '. scalar(misc::gmtime2($t)).' ' : '').
				($lu ? 'Last used '.misc::time_ago($lu).' ago' : '') if ($t or $lu);
		}
		$get_acc_list_mask->finish();
	} else {
		$get_acc_list->execute($cn, $level);
		while(my ($n, $a, $t, $lu, $id, $vh) = $get_acc_list->fetchrow_array) {
			push @reply, "*) $n ($id\@$vh)" . ($a ? ' Added by: '.$a : '');
			push @reply, '      '.($t ? 'Date/time added: '. scalar(misc::gmtime2($t)).' ' : '').
				($lu ? 'Last used '.misc::time_ago($lu).' ago' : '') if ($t or $lu);
		}
		$get_acc_list->finish();
	}

	notice($user, "$levels[$level] list for \002$cn\002:", @reply);

	return;
}

sub cs_xop_wipe($$$) {
	my ($user, $chan, $cmd, $nick) = @_;
	chk_registered($user, $chan) or return;
	
	my $slevel = get_best_acc($user, $chan);
	my $level = xop_byname($cmd);

	unless($level < $slevel) {
		notice($user, $err_deny);
		return;
	}
	can_do($chan, 'ACCCHANGE', $slevel, $user) or return;

	my $cn = $chan->{CHAN};

	$wipe_acc_list->execute($cn, $level);

	my $log_str = "wiped the $cmd list of \002$cn\002.";
	my $src = get_user_nick($user);
	notice($user, "You have $log_str");
	net::notice(agent($chan), '%'.$cn, "\002$src\002 has $log_str")
		if chk_flag($chan, F_VERBOSE);
	services::ulog($csnick, log::INFO(), $log_str, $user, $chan);
}

sub cs_xop_add($$$$) {
	my ($user, $chan, $cmd, $nick) = @_;
	
	chk_registered($user, $chan) or return;
	my $level = xop_byname($cmd);
	my $old = cs_xop_ad_pre($user, $chan, $nick, $level, 0);
	return unless defined($old);

	my $cn = $chan->{CHAN};
	
	if($old == $level) {
		notice($user, "\002$nick\002 already has $levels[$level] access to \002$cn\002.");
		return;
	}

	if($old == FOUNDER) {
		notice($user, "\002$nick\002 is the founder of \002$cn\002 and cannot be added to access lists.",
			"For more information, type: \002/msg chanserv help set founder\002");
		return;
	}

	my $root = get_root_nick($nick);
	my $auth = nickserv::chk_flag($root, nickserv::F_AUTH());
	my $src = get_user_nick($user);

	if($auth) {
		$add_auth->execute($cn, "$src:".($old ? $old : 0 ).":$level:".time(), $root);
		del_acc($root, $chan) if $level < $old;
	}
	else {
		set_acc($root, $user, $chan, $level);
	}

	if($old < 0) {
		$del_nick_akick->execute($cn, $root);
		my $log_str = "moved $root from the AKICK list to the ${levels[$level]} list of \002$cn\002".
			($auth ? ' (requires authorization)' : '');
			
		my $src = get_user_nick($user);
		notice_all_nicks($user, $root, "\002$src\002 $log_str");
		net::notice(agent($chan), '%'.$cn, "\002$src\002 $log_str")
			if chk_flag($chan, F_VERBOSE);
		services::ulog($csnick, log::INFO(), $log_str, $user, $chan);
	} else {
		my $log_str = ($old?'moved':'added')." \002$root\002" 
			. ($old ? " from the ${levels[$old]}" : '') .
			" to the ${levels[$level]} list of \002$cn\002" .
			($auth ? ' (requires authorization)' : '');
		my $src = get_user_nick($user);
		notice_all_nicks($user, $root, "\002$src\002 $log_str");
		net::notice(agent($chan), '%'.$cn, "\002$src\002 $log_str")
			if chk_flag($chan, F_VERBOSE);
		services::ulog($csnick, log::INFO(), $log_str, $user, $chan);
	}
}

sub cs_xop_del($$$) {
	my ($user, $chan, $cmd, $nick) = @_;

	chk_registered($user, $chan) or return;
	my $level = xop_byname($cmd);
	my $old = cs_xop_ad_pre($user, $chan, $nick, $level, 1);
	return unless defined($old);

	my $cn = $chan->{CHAN};
	
	unless($old == $level) {
		notice($user, "\002$nick\002 is not on the ${levels[$level]} list of \002$cn\002.");
		return;
	}

	my $root = get_root_nick($nick);

	del_acc($root, $chan);

	my $src = get_user_nick($user);
	my $log_str = "removed \002$root\002 ($nick) from the ${levels[$level]} list of \002$cn\002";
	notice_all_nicks($user, $root, "\002$src\002 $log_str");
	net::notice(agent($chan), '%'.$cn, "\002$src\002 $log_str")
		if chk_flag($chan, F_VERBOSE);
	services::ulog($csnick, log::INFO(), $log_str, $user, $chan);
}

sub cs_count($$) {
   	my ($user, $chan) = @_;
	
	chk_registered($user, $chan) or return;
	
	can_do($chan, 'ACCLIST', undef, $user) or return;

	my $cn = $chan->{CHAN};
	
	my $reply = '';
	for (my $level = $plzero + 1; $level < COFOUNDER + 2; $level++) {
		$get_acc_count->execute($cn, $level - 1);
		my ($num_recs) = $get_acc_count->fetchrow_array;
		$reply = $reply." $plevels[$level]: ".$num_recs;
	}
	notice($user, "\002$cn Count:\002 ".$reply);
}

sub cs_levels_pre($$;$) {
	my($user, $chan, $listonly) = @_;

	chk_registered($user, $chan) or return 0;

	return can_do($chan, ($listonly ? 'LEVELSLIST' : 'LEVELS'), undef, $user);
}

sub cs_levels_set($$$;$) {
	my ($user, $chan, $perm, $level) = @_;

	cs_levels_pre($user, $chan) or return;
	my $cn = $chan->{CHAN};

	unless(is_level($perm)) {
		notice($user, "$perm is not a valid permission.");
		return;
	}

	if(defined($level)) {
		$level = xop_byname($level);
		unless(defined($level) and $level >= 0) {
			notice($user, 'You must specify one of the following levels: '.
				'any, uop, vop, hop, aop, sop, cofounder, founder, nobody');
			return;
		}

		$get_level_max->execute($perm);
		my ($max) = $get_level_max->fetchrow_array;
		$get_level_max->finish();

		if($max and $level > $max) {
			notice($user, "\002$perm\002 cannot be set to " . $plevels[$level+$plzero] . '.');
			return;
		}
		
		$add_level->execute($cn, $perm);
		$set_level->execute($level, $cn, $perm);
		
		if($level == 8) {
			notice($user, "\002$perm\002 is now disabled in \002$cn\002.");
		} else {
			notice($user, "\002$perm\002 now requires " . $levels[$level] . " access in \002$cn\002.");
		}
	} else {
		$reset_level->execute($perm, $cn);

		notice($user, "\002$perm\002 has been reset to default.");
	}
}

sub cs_levels_list($$) {
	my ($user, $chan) = @_;

	cs_levels_pre($user, $chan, 1) or return;
	my $cn = $chan->{CHAN};

	$get_levels->execute($cn);
	my @data;
	while(my ($name, $def, $lvl) = $get_levels->fetchrow_array) {
		push @data, [$name,
			(defined($lvl) ? $plevels[$lvl+$plzero] : $plevels[$def+$plzero]),
			(defined($lvl) ? '' : '(default)')];
	}

	notice($user,
		"Permission levels for \002$cn\002:",
		fmt::columnar(@data)
	);
}

sub cs_levels_clear($$) {
	my ($user, $chan) = @_;

	cs_levels_pre($user, $chan) or return;
	my $cn = $chan->{CHAN};

	$clear_levels->execute($cn);

	notice($user, "All permissions have been reset to default.");
}

sub cs_akick_pre($$;$) {
	my ($user, $chan, $list) = @_;
	
	chk_registered($user, $chan) or return 0;

	return can_do($chan, ($list ? 'AKICKLIST' : 'AKICK'), undef, $user);
}

sub cs_akick_add($$$$) {
	my ($user, $chan, $mask, $reason) = @_;
	my $cn = $chan->{CHAN};

	my $adder = cs_akick_pre($user, $chan) or return;
	
	my ($nick, $ident, $host) = misc::parse_mask($mask);

	if(($ident eq '' or $host eq '') and not ($ident eq '' and $host eq '')) {
		notice($user, 'Invalid hostmask.');
		return;
	}

	if($ident eq '') {
		$nick = $mask;
		
		unless($nick =~ $valid_nick_re) {
			notice($user, 'Invalid nickname.');
			return;
		}
	}

	if ($ident eq '' and $host eq '' and !nickserv::is_registered($nick)) {
		notice($user, "\002$nick\002 is not registered");
		return;
	}

	my $rlength = length($reason);
	if($rlength >= 350) {
		notice($user, 'AKick reason is too long by '. $rlength-350 .' character(s). Maximum length is 350 characters.');
		return;
	}

	my $log_str;
	my $src = get_user_nick($user);
	if($ident eq '' and $host eq '' and my $old = get_acc($nick, $chan)) {
		if ($old == -1) {
			notice($user, "\002$nick\002 is already on the AKick list in \002$cn\002");
			return;
		}
		if($old < get_best_acc($user, $chan) or adminserv::can_do($user, 'SERVOP')) {
			if ($old == FOUNDER()) {
			# This is a fallthrough for the override case.
			# It shouldn't happen otherwise.
			# I didn't make it part of the previous conditional
			# b/c just $err_deny is a bit undescriptive in the override case.
				notice($user, "You can't akick the founder!", $err_deny);
				return;
			}
			
			my $root = get_root_nick($nick);
			set_acc($nick, $user, $chan, -1);
			$add_nick_akick->execute($cn, $src, $reason, $nick); $add_nick_akick->finish();
			$log_str = "moved \002$nick\002 (root: \002$root\002) from the $levels[$old] list".
				" to the AKick list of \002$cn\002";
		} else {
			notice($user, $err_deny);
			return;
		}
	} else {
		if($ident eq '' and $host eq '') {
			$add_nick_akick->execute($cn, $src, $reason, $nick); $add_nick_akick->finish();
			if (find_auth($cn, $nick)) { 
			# Don't allow a pending AUTH entry to potentially override an AKick entry
			# Believe it or not, it almost happened with #animechat on SCnet.
			# This would also end up leaving an orphan entry in the akick table.
				$nickserv::del_auth->execute($nick, $cn);
				$nickserv::del_auth->finish();
			}
			set_acc($nick, $user, $chan, -1);
			my $root = get_root_nick($nick);
			$log_str = "added \002$nick\002 (root: \002$root\002) to the AKick list of \002$cn\002.";
		} else {
			($nick, $ident, $host) = misc::glob2sql($nick, $ident, $host);
			unless($add_akick->execute($cn, $nick, $ident, $host, $adder, $reason)) {
				notice($user, "\002$mask\002 is already on the AKick list of \002$cn\002.");
				return;
			}
			$log_str = "added \002$mask\002 to the AKick list of \002$cn\002.";
		}
		
	}
	notice($user, "You have $log_str");
	net::notice(agent($chan), '%'.$cn, "\002$src\002 $log_str")
		if chk_flag($chan, F_VERBOSE);
	services::ulog($csnick, log::INFO(), $log_str, $user, $chan);

	akick_allchan($chan);
}

sub cs_akick_del($$$) {
	my ($user, $chan, $mask) = @_;
	my $cn = $chan->{CHAN};

	cs_akick_pre($user, $chan) or return;

	my ($nick, $ident, $host) = misc::parse_mask($mask);

	if(($ident eq '' or $host eq '') and not ($ident eq '' and $host eq '')) {
		notice($user, 'Invalid hostmask.');
		return;
	}

	my $log_str;
	if($ident eq '' and $host eq '') {
		if(get_acc($nick, $chan) == -1) {
			del_acc($nick, $chan);
			$del_nick_akick->execute($cn, $nick); $del_nick_akick->finish();
			my $root = get_root_nick($nick);
			$log_str = "deleted \002$nick\002 (root: \002$root\002) from the AKick list of \002$cn\002.";
		} else {
			notice($user, "\002$mask\002 was not on the AKick list of \002$cn\002.");
			return;
		}
	} else {
		($nick, $ident, $host) = misc::glob2sql($nick, $ident, $host);
		if($del_akick->execute($cn, $nick, $ident, $host) != 0) {
			$log_str = "deleted \002$mask\002 from the AKick list of \002$cn\002.";
		} else {
			notice($user, "\002$mask\002 was not on the AKick list of \002$cn\002.");
			return;
		}
	}
	my $src = get_user_nick($user);
	notice($user, "\002$src\002 $log_str");
	services::ulog($csnick, log::INFO(), $log_str, $user, $chan);
	net::notice(agent($chan), '%'.$cn, "\002$src\002 $log_str") if chk_flag($chan, F_VERBOSE);
}

sub cs_akick_list($$) {
	my ($user, $chan) = @_;
	my $cn = $chan->{CHAN};

	cs_akick_pre($user, $chan, 1) or return;

	my @data;
	
	$get_akick_list->execute($cn);
	while(my ($nick, $ident, $host, $adder, $reason, $time) = $get_akick_list->fetchrow_array) {
		if($ident ne '') {
			($nick, $ident, $host) = misc::sql2glob($nick, $ident, $host);
		}

		if($ident eq '' and $host eq '') {
			$nick = nickserv::get_id_nick($nick);
		} else {
			$nick = "$nick!$ident\@$host";
		}

		push @data, [$nick, $adder, ( $time ? misc::gmtime2($time) : '' ), $reason];
	}

	notice($user,
		"AKICK list of \002$cn\002:",
		fmt::columnar(@data)
	);
}

sub cs_akick_wipe($$$) {
	my ($user, $chan) = @_;
	my $cn = $chan->{CHAN};

	cs_akick_pre($user, $chan) or return;

	$drop_akick->execute($cn);
	my $log_str = "wiped the AKICK list of \002$cn\002.";
	my $src = get_user_nick($user);
	notice($user, "You have $log_str");
	net::notice(agent($chan), '%'.$cn, "\002$src\002 $log_str") if chk_flag($chan, F_VERBOSE);
	services::ulog($csnick, log::INFO(), $log_str, $user, $chan);
}

sub cs_akick_enforce($$$) {
	my ($user, $chan) = @_;
	my $cn = $chan->{CHAN};

	chk_registered($user, $chan) or return;

	can_do($chan, 'AKickEnforce', undef, $user) or return;

	akick_allchan($chan);
}

sub cs_info($$) {
	my($user, $chan) = @_;
	my $cn = $chan->{CHAN};
	
	unless(can_do($chan, 'INFO', 0, undef, 1)) {
		can_do($chan, 'INFO', undef, $user) or return;
	}

	$get_info->execute($cn);
	my @result = $get_info->fetchrow_array;
	unless(@result) { notice($user, "The channel \002$cn\002 is not registered."); return; }

	my ($descrip, $regd, $last, $topic, $topicer, $modelock, $founder, $successor, $bot, $bantype) = @result;

	$modelock = modes::sanitize($modelock) unless can_do($chan, 'GETKEY', undef, $user, 1);

	my @opts;

	my $topiclock = get_level($chan, 'SETTOPIC');
	push @opts, "Topic Lock ($levels[$topiclock])" if $topiclock;
	
	if(chk_flag($chan, (F_CLOSE | F_DRONE))) {
		notice($user, "\002$cn\002 is closed and cannot be used: ". get_close($chan));
		return;
	}

	push @opts, 'Will not expire' if chk_flag($chan, F_HOLD);
	push @opts, 'Channel is frozen and access suspended' if chk_flag($chan, F_FREEZE);
	push @opts, 'OpGuard enabled' if chk_flag($chan, F_OPGUARD);
	push @opts, 'Channel bot will stay in channel' if chk_flag($chan, F_BOTSTAY);
	push @opts, 'ChanOp status may be kept after a netsplit' if chk_flag($chan, F_SPLITOPS);
	push @opts, 'Verbose mode enabled' if chk_flag($chan, F_VERBOSE);
	#push @opts, 'Per user greets enabled' if chk_flag($chan, F_GREET);
	push @opts, 'Ban-type: '.$bantype if $bantype;
	my $opts = join(', ', @opts);

	my (@reply, @data);
	
	push @reply, "ChanServ info for \002$cn\002:";
	push @data,	['Founder:', $founder];
	push @data, 	['Successor:', $successor] if $successor;
	push @data, 	['Description:', $descrip] if $descrip;
	push @data,	['Mode lock:',	$modelock];
	push @data, 	['Options:',	$opts] if $opts;
	push @data,	['ChanBot:',	$bot] if $bot and $bot ne '';
	#memo level
	push @data,	['Registered:', scalar(misc::gmtime2($regd))],
			['Last opping:', scalar(misc::gmtime2($last))],
			['Time now:', scalar(misc::gmtime2())];
	#push @reply, " --";
	push @reply, fmt::columnar(@data);
	notice($user, @reply);
}

sub cs_set_pre($$$$) {
	my ($user, $chan, $set, $parm) = @_;
	my $cn = $chan->{CHAN};
	my $override = 0;
	
	chk_registered($user, $chan) or return 0;


	if(
	   ($set =~ /^(hold|no-?expire|log)$/i and not adminserv::can_do($user, 'SERVOP')) or 
	   ($set eq 'freeze' and not adminserv::can_do($user, 'FREEZE')) or
	   ($set eq 'botstay' and not adminserv::can_do($user, 'BOT'))
	  ) {
		notice($user, $err_deny);
		return 0;
	} else {
		$override = 1;
	}
	unless ($override) {
		can_do($chan, 'SET', undef, $user) or return 0;
	}
	
	unless(
		$set eq 'founder' or
		$set eq 'successor' or
		$set eq 'unsuccessor' or
		$set =~ /m(?:ode)?lock/ or
		$set eq 'desc' or
		$set eq 'topiclock' or
		#$set eq 'welcome' or $set eq 'entrymsg' or
		$set eq 'greet' or
		
		$set eq 'opguard' or $set eq 'secureops' or
		$set =~ /^(hold|no-?expire)$/i or $set eq 'freeze' or
		$set eq 'botstay' or $set eq 'verbose' or
		$set eq 'splitops' or $set eq 'bantype' or
		$set eq 'dice' or $set eq 'welcomeinchan' or
		$set eq 'log' or $set =~ /^a(?:uto)?voice$/i
	) {
		notice($user, "$set is not a valid ChanServ setting.");
		return 0;
	}
	return 1;
}

sub cs_set($$$;$) {
	my ($user, $chan, $set, $parm) = @_;
	my $cn = $chan->{CHAN};
	$set = lc $set;

	cs_set_pre($user, $chan, $set, $parm) or return;

	if($set =~ /^founder$/i) {
		my $override;
		unless(get_best_acc($user, $chan) == FOUNDER) {
			if(adminserv::can_do($user, 'SERVOP')) {
				$override = 1;
			} else {
				notice($user, $err_deny);
				return;
			}
		}

		my $root;
		unless($root = get_root_nick($parm)) {
			notice($user, "The nick \002$parm\002 is not registered.");
			return;
		}
		
		$get_founder->execute($cn);
		my ($prev) = $get_founder->fetchrow_array;
		$get_founder->finish();

		if(lc($root) eq lc($prev)) {
			notice($user, "\002$parm\002 is already the founder of \002$cn\002.");
			return;
		}
		
		set_acc($prev, $user, $chan, COFOUNDER);

		$set_founder->execute($root, $cn); $set_founder->finish();
		set_acc($root, $user, $chan, FOUNDER);

		notice($user, ($override ? "The previous founder, \002$prev\002, has" : "You have") . " been moved to the co-founder list of \002$cn\002.");
		notice_all_nicks($user, $root, "\002$root\002 has been set as the founder of \002$cn\002.");
		services::ulog($csnick, log::INFO(), "set founder of \002$cn\002 to \002$root\002", $user, $chan);

		$get_successor->execute($cn);
		my $suc = $get_successor->fetchrow_array; $get_successor->finish();
		if(lc($suc) eq lc($root)) {
			$del_successor->execute($cn); $del_successor->finish();
			notice($user, "Successor has been removed from \002$cn\002.");
		}

		return;
	}

	if($set eq 'successor') {
		unless(get_best_acc($user, $chan) == FOUNDER or adminserv::can_do($user, 'SERVOP')) {
			notice($user, $err_deny);
			return;
		}

		if(get_acc($parm, $chan) == 7) {
			notice($user, "The channel founder may not be the successor.");
			return;
		}

		my $root;
		unless($root = get_root_nick($parm)) {
			notice($user, "The nick \002$parm\002 is not registered.");
			return;
		}

		$set_successor->execute($root, $cn); $set_successor->finish();

		notice($user, "\002$parm\002 is now the successor of \002$cn\002");
		services::ulog($csnick, log::INFO(), "set successor of \002$cn\002 to \002$root\002", $user, $chan);
		return;
	}

	if($set eq 'unsuccessor') {
		unless(get_best_acc($user, $chan) == FOUNDER or adminserv::can_do($user, 'SERVOP')) {
			notice($user, $err_deny);
			return;
		}

		$del_successor->execute($cn); $del_successor->finish();

		notice($user, "Successor has been removed from \002$cn\002.");
		services::ulog($csnick, log::INFO(), "removed successor from \002$cn\002", $user, $chan);
		return;
	}

	if($set =~ /m(?:ode)?lock/) {
		$parm = modes::merge($parm, '+r', 1);
		$set_modelock->execute($parm, $cn);

		notice($user, "Mode lock for \002$cn\002 has been set to: \002$parm\002");
		do_modelock($chan);
		return;
	}

	if($set eq 'desc') {
		$set_descrip->execute($parm, $cn);

		notice($user, "Description of \002$cn\002 has been changed.");
		return;
	}

	if($set eq 'topiclock') {
		my $perm = xop_byname($parm);
		if($parm =~ /^(?:no|off|false|0)$/i) {
			cs_levels_set($user, $chan, 'SETTOPIC');
			cs_levels_set($user, $chan, 'TOPIC');
		} elsif($perm >= 0 and defined($perm)) {
			cs_levels_set($user, $chan, 'SETTOPIC', $parm);
			cs_levels_set($user, $chan, 'TOPIC', $parm);
		} else {
			notice($user, 'Syntax: SET <#chan> TOPICLOCK <off|any|uop|vop|hop|aop|sop|cf|founder>');
		}
		return;
	}

	if($set =~ /^bantype$/i) {
		unless (misc::isint($parm) and $parm >= 0 and $parm <= 9) {
			notice($user, 'Invalid bantype');
			return;
		}

		$set_bantype->execute($parm, $cn);

		notice($user, "Ban-Type for \002$cn\002 now set to \002$parm\002.");

		return;
	}
	
	my $val;
	if($parm =~ /^(?:no|off|false|0)$/i) { $val = 0; }
	elsif($parm =~ /^(?:yes|on|true|1)$/i) { $val = 1; }
	else {
		notice($user, "Please say \002on\002 or \002off\002.");
		return;
	}
	
	if($set =~ /^(?:opguard|secureops)$/i) {
		set_flag($chan, F_OPGUARD, $val);

		if($val) {
			notice($user,
				"OpGuard is now \002ON\002.",
				"Channel status may not be granted by unauthorized users in \002$cn\002."#,
				#"Note that you must change the $csnick LEVELS settings for VOICE, HALFOP, OP, and/or ADMIN for this setting to have any effect."
			);
		} else {
			notice($user,
				"OpGuard is now \002OFF\002.",
				"Channel status may be given freely in \002$cn\002."
			);
		}

		return;
	}

	if($set =~ /^(?:splitops)$/i) {
		set_flag($chan, F_SPLITOPS, $val);

		if($val) {
			notice($user, "SplitOps is now \002ON\002.");
		} else {
			notice($user, "SplitOps is now \002OFF\002.");
		}

		return;
	}

	if($set =~ /^(hold|no-?expire)$/i) {
		set_flag($chan, F_HOLD, $val);

		if($val) {
			notice($user, "\002$cn\002 will not expire");
			services::ulog($csnick, log::INFO(), "has held \002$cn\002", $user, $chan);
		} else {
			notice($user, "\002$cn\002 is no longer held from expiration");
			services::ulog($csnick, log::INFO(), "has removed \002$cn\002 from hold", $user, $chan);
		}

		return;
	}

	if($set =~ /^freeze$/i) {
		set_flag($chan, F_FREEZE, $val);

		if($val) {
			notice($user, "\002$cn\002 is now frozen and access suspended");
			services::ulog($csnick, log::INFO(), "has frozen \002$cn\002", $user, $chan);
		} else {
			notice($user, "\002$cn\002 is now unfrozen and access restored");
			services::ulog($csnick, log::INFO(), "has unfrozen \002$cn\002", $user, $chan);
		}

		return;
	}

	if($set =~ /^botstay$/i) {
		set_flag($chan, F_BOTSTAY, $val);

		if($val) {
			notice($user, "Bot will now always stay in \002$cn");
			botserv::bot_join($chan, undef);
		} else {
			notice($user, "Bot will now part if less than one user is in \002$cn");
			botserv::bot_part_if_needed(undef, $chan, "Botstay turned off");
		}

		return;
	}
	if($set =~ /^verbose$/i) {
		set_flag($chan, F_VERBOSE, $val);

		if($val) {
			notice($user, "Verbose mode enabled on \002$cn");
		}
		else {
			notice($user, "Verbose mode disabled on \002$cn");
		}
		return;
	}

	if($set =~ /^greet$/i) {
		if($val) {
			notice($user, "$csnick SET $cn GREET ON is deprecated.", 
				"Please use $csnick LEVELS $cn SET GREET <rank>");
		} else {
			cs_levels_set($user, $chan, 'GREET', 'nobody');
		}

		return;
	}

	if($set =~ /^dice$/i) {
		if($val) {
			notice($user, "$csnick SET $cn DICE ON is deprecated.", 
				"Please use $csnick LEVELS $cn SET DICE <rank>");
		} else {
			cs_levels_set($user, $chan, 'DICE', 'nobody');
		}

		return;
	}

	if($set =~ /^welcomeinchan$/i) {
		set_flag($chan, F_WELCOMEINCHAN(), $val);

		if($val) {
			notice($user, "WELCOME messages will be sent privately.");
		} else {
			notice($user, "WELCOME messages will be put in the channel.");
		}

		return;
	}

	if($set =~ /^log$/i) {
		unless(module::is_loaded('logserv')) {
			notice($user, "module logserv is not loaded, logging is not available.");
			return;
		}

		if($val) {
			logserv::addchan($user, $cn) and set_flag($chan, F_LOG, $val);
			#notice($user, "Verbose mode enabled on \002$cn");
		}
		else {
			logserv::delchan($user, $cn) and set_flag($chan, F_LOG, $val);
			#notice($user, "Verbose mode disabled on \002$cn");
		}
		return;
	}

	if($set =~ /^a(?:uto)?voice$/i) {
		set_flag($chan, F_AUTOVOICE(), $val);

		if($val) {
			notice($user, "All users w/o access will be autovoiced on join.");
		} else {
			notice($user, "AUTOVOICE disabled.");
		}

		return;
	}
}

sub cs_why($$@) {
	my ($user, $chan, @tnicks) = @_;

	chk_registered($user, $chan) or return;

	can_do($chan, 'ACCLIST', undef, $user) or return;

	my $cn = $chan->{CHAN};
	my @reply;
	foreach my $tnick (@tnicks) {
		my $tuser = { NICK => $tnick };
		unless(get_user_id($tuser)) {
			notice($user, "\002$tnick\002: No such user.");
			return;
		}

		my $has;
		if(is_online($tnick)) {
			$has = 'has';
		} else {
			$has = 'had';
		}

		my $n;
		$get_all_acc->execute(get_user_id($tuser), $cn);
		while(my ($rnick, $acc) = $get_all_acc->fetchrow_array) {
			$n++;
			push @reply, "\002$tnick\002 $has $plevels[$acc+$plzero] access to \002$cn\002 due to identification to the nick \002$rnick\002.";
		}
		$get_all_acc->finish();

		unless($n) {
			push @reply, "\002$tnick\002 has no access to \002$cn\002.";
		}
	}
	notice($user, @reply);
}

sub cs_setmodes($$$@) {
	my ($user, $cmd, $chan, @args) = @_;
	my $agent = $user->{AGENT} or $csnick;
	my $src = get_user_nick($user);
	my $cn = $chan->{CHAN};
	my $self;
	
	if (chk_flag($chan, F_FREEZE())) {
		notice($user, "\002$cn\002 is frozen and access suspended.");
		return;
	}
	
	if(scalar(@args) == 0) {
		@args = ($src);
		$self = 1;
	} elsif((scalar(@args) == 1) and (lc($args[0]) eq lc($src))) {
		$self = 1;
	}

	# PROTECT is deprecated. remove it in a couple versions.
	# It should be called ADMIN under PREFIX_AQ
	my @mperms = ('VOICE', 'HALFOP', 'OP', 'ADMIN');
	my @l = ('v', 'h', 'o', 'a');
	my ($level, @modes, $count);
	
	if($cmd =~ /voice$/i) { $level = 0 }
	elsif($cmd =~ /h(alf)?op$/i) { $level = 1 }
	elsif($cmd =~ /op$/i) { $level = 2 }
	elsif($cmd =~ /(protect|admin)$/i) { $level = 3 }
	elsif($cmd =~ /owner$/i) { $level = 4 }
	my $de = 1 if($cmd =~ s/^de//i);
	#$cmd =~ s/^de//i;

	my $acc = get_best_acc($user, $chan);
	
	# XXX I'm not sure this is the best way to do it.
	print "ACC: $acc, $level";
	unless(($de and $self) or ($self and ($level + 2) <= $acc) or can_do($chan, $mperms[$level], $acc, $user, 1)) {
		notice($user, $err_deny);
		return;
	}

	my ($override, $check_override);

	foreach my $target (@args) {
		my ($tuser);
		
		$tuser = ($self ? $user : { NICK => $target } );
		
		unless(is_in_chan($tuser, $chan)) {
			notice($user, "\002$target\002 is not in \002$cn\002.");
			next;
		}
		
		my $top = get_op($tuser, $chan);
		
		if($de) {
			unless($top & (2**$level)) {
				notice($user, "\002$target\002 has no $cmd in \002$cn\002.");
				next;
			}
			
			if(!$override and get_best_acc($tuser, $chan) > $acc) {
				unless($check_override) {
					$override = adminserv::can_do($user, 'SUPER');
					$check_override = 1;
				}
				if($check_override and !$override) {
					notice($user, "\002$target\002 outranks you in \002$cn\002.");
					next;
				}
			}
		} else {
			if($top & (2**$level)) {
				notice($user, "\002$target\002 already has $cmd in \002$cn\002.");
				next;
			}
			if (chk_flag($chan, F_OPGUARD()) and
				!can_keep_op($user, $chan, $tuser, $l[$level]))
			{
				notice($user, "$target may not hold ops in $cn because OpGuard is enabled. ".
					"Please respect the founders wishes.");
				next;
			}
		}

		push @modes, [($de ? '-' : '+').$l[$level], $target];
		$count++;

	}

	net::setmode2(agent($chan), $cn, @modes) if scalar @modes;
	net::notice(agent($chan), '%'.$cn, "$src used ".($de ? "de$cmd" : $cmd).' '.join(' ', @args))
		if !$self and (lc $user->{AGENT} eq lc $csnick) and chk_flag($chan, F_VERBOSE);
}


sub cs_drop($$) {
	my ($user, $chan) = @_;
	my $cn = $chan->{CHAN};

	chk_registered($user, $chan) or return;

	unless(get_best_acc($user, $chan) == FOUNDER or adminserv::can_do($user, 'SERVOP')) {
		notice($user, $err_deny);
		return;
	}

	drop($chan);
	notice($user, $cn.' has been dropped.');
	services::ulog($csnick, log::INFO(), "dropped $cn", $user, $chan);

	undef($enforcers{lc $cn});
	botserv::bot_part_if_needed(undef(), $chan, "Channel dropped.");
}

sub cs_kick($$$;$$) {
	my ($user, $chan, $target, $ban, $reason) = @_;
	
	my $srclevel = get_best_acc($user, $chan);

	my ($nick, $override) = can_do($chan, ($ban ? 'BAN' : 'KICK'), $srclevel, $user);
	return unless $nick;

	my $src = get_user_nick($user);
	my $cn = $chan->{CHAN};

	$reason = "Requested by $src".($reason?": $reason":'');
	
	my @errors = (
		["I'm sorry, $src, I'm afraid I can't do that."],
		["They are not in \002$cn\002."],
		[$err_deny],
		["User not found"],
	);
	my @notinchan = ();
	my $peace = ({modes::splitmodes(get_modelock($chan))}->{Q}->[0] eq '+');
	
	my @targets = split(/\,/, $target);
	foreach $target (@targets) {
		my $tuser = { NICK => $target };
		my $targetlevel = get_best_acc($tuser, $chan);

		if(lc $target eq lc agent($chan) or adminserv::is_service($tuser)) {
			push @{$errors[0]}, $target;
			next;
		}
		
		if(get_user_id($tuser)) {
			unless(is_in_chan($tuser, $chan)) {
				if ($ban) {
					push @notinchan, $tuser;
				} else {
					push @{$errors[1]}, $target;
				}
				next;
			}
		} else {
			push @{$errors[3]}, $target;
			next;
		}

		
		if( ( ($peace and $targetlevel) or ($srclevel <= $targetlevel) ) and not $override) {
			push @{$errors[2]}, $target;
			next;
		}
	
		if($ban) {
			kickban($chan, $tuser, undef, $reason);
		} else {
			net::kick(agent($chan), $cn, $target, $reason) unless adminserv::is_service($user);
		}
	}
	
	foreach my $errlist (@errors) {
		if(@$errlist > 1) {
			my $msg = shift @$errlist;
			
			foreach my $e (@$errlist) { $e = "\002$e\002" }
			
			notice($user,
				"Cannot kick ".
				fmt::enum("or", @$errlist).
				": $msg"
			);
		}
	}
	cs_ban($user, $chan, @notinchan) if ($ban and scalar (@notinchan));
}

sub cs_kickmask($$$;$$) {
	my ($user, $chan, $mask, $ban, $reason) = @_;

	my $srclevel = get_best_acc($user, $chan);

	my ($nick, $override) = can_do($chan, ($ban ? 'BAN' : 'KICK'), $srclevel, $user);

	my $src = get_user_nick($user);
	my $cn = $chan->{CHAN};

	$reason = "Requested by $src".($reason?": $reason":'');

	my $count = kickmask_noacc($chan, $mask, $reason, $ban);
	#notice($user, ($count ? "Users kicked from \002$cn\002: $count." : "No users in \002$cn\002 matched $mask."))
	notice($user, "No users in \002$cn\002 matched $mask.")
		unless $mask;
}

sub cs_ban($$@) {
	my ($user, $chan, @targets) = @_;
	my $cn = $chan->{CHAN};
	my $src = get_user_nick($user);

	my $srclevel = get_best_acc($user, $chan);
	my ($nick, $override) = can_do($chan, 'BAN', $srclevel, $user);
	return unless $nick;

	my @errors = (
		["I'm sorry, $src, I'm afraid I can't do that."],
		["User not found"],
		[$err_deny]
	);

	my @bans;
	foreach my $target (@targets) {
		my $tuser;

		if(ref($target)) {
			$tuser = $target;
		} 
		elsif($target =~ /\,/) {
			push @targets, split(',', $target);
			next;
		}
		elsif($target eq '') {
			# Should never happen
			# but it could, given the split above
			next;
		}
		elsif($target =~ /.+!.+\@.+/) {
			push @bans, $target;
			next;
		}
		else {
			$tuser = { NICK => $target };
		}

		my $targetlevel = get_best_acc($tuser, $chan);

		if(lc $target eq lc agent($chan) or adminserv::is_service($tuser)) {
			push @{$errors[0]}, get_user_nick($tuser);
			next;
		}
		
		unless(get_user_id($tuser)) {
			push @{$errors[1]}, get_user_nick($tuser);
			next;
		}
		if($srclevel <= $targetlevel and not $override) {
			push @{$errors[2]}, $target;
			next;
		}

		push @bans, make_banmask($chan, $tuser);
	}

	foreach my $errlist (@errors) {
		if(@$errlist > 1) {
			my $msg = shift @$errlist;
			
			foreach my $e (@$errlist) { $e = "\002$e\002" }
			
			notice($user,
				"Cannot ban ".
				fmt::enum("or", @$errlist).
				": $msg"
			);
		}
	}

	net::ban_list(agent($chan), $cn, +1, 'b', @bans) if (scalar(@bans));
	net::notice(agent($chan), $cn, "$src used BAN ".join(' ', @bans))
		if (lc $user->{AGENT} eq lc $csnick) and (chk_flag($chan, F_VERBOSE) and scalar(@bans));

}

sub cs_invite($$@) {
	my ($user, $chan, @targets) = @_;
	my $src = get_user_nick($user);
	my $cn = $chan->{CHAN};
	my $srclevel = get_best_acc($user, $chan);

	my @errors = (
		["They are not online."],
		["They are already in \002$cn\002."],
		[$err_deny]
	);

	my @invited;
	foreach my $target (@targets) {
		my $tuser;
		if(lc($src) eq lc($target)) {
			$tuser = $user;
		}
		elsif($target =~ /\,/) {
			push @targets, split(',', $target);
			next;
		}
		elsif($target eq '') {
			# Should never happen
			# but it could, given the split above
			next;
		}
		else {
			$tuser = { NICK => $target };
		}

		if(lc($src) eq lc($target)) {
			unless(can_do($chan, 'InviteSelf', $srclevel, $user, 1)) {
				push @{$errors[2]}, $target;
				next;
			}
		}
		else {
			unless(can_do($chan, 'INVITE', $srclevel, $user, 1)) {
				push @{$errors[2]}, $target;
				next;
			}
			
			unless(nickserv::is_online($target)) {
				push @{$errors[0]}, $target;
				next;
			}
			
			# invite is annoying, so punish them mercilessly
			return if operserv::flood_check($user, 2);
		}
		
		if(is_in_chan($tuser, $chan)) {
			push @{$errors[1]}, $target;
			next;
		}
		
		net::invite(agent($chan), $cn, $target); push @invited, $target;
		net::notice(agent($chan), $target, "\002$src\002 has invited you to \002$cn\002.") unless(lc($src) eq lc($target));
	}

	foreach my $errlist (@errors) {
		if(@$errlist > 1) {
			my $msg = shift @$errlist;
			
			foreach my $e (@$errlist) { $e = "\002$e\002" }
			
			notice($user,
				"Cannot invite ".
				fmt::enum("or", @$errlist).
				": $msg"
			);
		}
	}
	
	net::notice(agent($chan), $cn, "$src used INVITE ".join(' ', @invited))
		if (lc $user->{AGENT} eq lc $csnick)and chk_flag($chan, F_VERBOSE) and scalar(@invited);
}

sub cs_close($$$) {
	my ($user, $chan, $reason, $type) = @_;
	# $type is a flag, either F_CLOSE or F_DRONE
	my $cn = $chan->{CHAN};
	my $oper;

	unless($oper = adminserv::is_svsop($user, adminserv::S_ROOT())) {
		notice($user, $err_deny);
		return;
	}

	my $rlength = length($reason);
	if($rlength >= 350) {
		notice($user, 'Close reason is too long by '. $rlength-350 .' character(s). Maximum length is 350 characters.');
		return;
	}

	if(is_registered($chan)) {
		$drop_acc->execute($cn);
		$drop_lvl->execute($cn);
		$del_close->execute($cn);
		$drop_akick->execute($cn);
		$drop_welcome->execute($cn);
		$drop_chantext->execute($cn);
		$drop_nicktext->execute($cn); # Leftover channel auths

		$set_founder->execute($oper, $cn);
	}
	else {
		$register->execute($cn, $reason, $oper);
	}
	$set_modelock->execute('+rsnt', $cn);
	do_modelock($chan);
	set_acc($oper, undef, $chan, FOUNDER);

	$set_close->execute($cn, $reason, $oper, $type);
	foreach my $f (F_FREEZE, F_CLOSE, F_DRONE) { set_flag($chan, $f, 0) }
	foreach my $f ($type, F_HOLD) { set_flag($chan, $f, 1) }

	my $src = get_user_nick($user);
	my $time = misc::gmtime2(time());
	my $cmsg = "is closed [$src $time]: $reason";

	if ($type == F_CLOSE) {
		clear_users($chan, "Channel $cmsg");
		net::settopic(agent($chan), $cn, $src, time(), "Channel $cmsg")
	}
	elsif ($type == F_DRONE) {
		chan_kill($chan, "$cn $cmsg");
	}

	notice($user, "The channel \002$cn\002 is now closed.");
	services::ulog($csnick, log::INFO(), "closed $cn with reason: $reason", $user, $chan);
}

sub cs_clear_pre($$) {
	my ($user, $chan) = @_;
	my $cn = $chan->{CHAN};

	my $srclevel = get_best_acc($user, $chan);

	my ($cando, $override) = can_do($chan, 'CLEAR', $srclevel, $user);
	return 0 unless($cando);

	$get_highrank->execute($cn);
	my ($highrank_nick, $highrank_level) = $get_highrank->fetchrow_array();
	$get_highrank->finish();

	if($highrank_level > $srclevel && !$override) {
		notice($user, "$highrank_nick outranks you in $cn (level: $levels[$highrank_level])");
		return 0;
	}

	return 1;
}

sub cs_clear_users($$;$) {
	my ($user, $chan, $reason) = @_;
	my $src = get_user_nick($user);

	cs_clear_pre($user, $chan) or return;

	my $rlength = length($reason);
	if($rlength >= 350) {
		notice($user, 'Clear reason is too long by '. $rlength-350 .' character(s). Maximum length is 350 characters.');
		return;
	}
	
	clear_users($chan, "CLEAR USERS by \002$src\002".($reason?" reason: $reason":''));
}

sub cs_clear_modes($$;$) {
	my ($user, $chan, $reason) = @_;
	my $cn = $chan->{CHAN};
	my $src = get_user_nick($user);

	cs_clear_pre($user, $chan) or return;

	my $rlength = length($reason);
	if($rlength >= 350) {
		notice($user, 'Clear reason is too long by '. $rlength-350 .' character(s). Maximum length is 350 characters.');
		return;
	}

	my $agent = agent($chan);
	net::notice($agent, $cn, "CLEAR MODES by \002$src\002".($reason?" reason: $reason":''));

	$get_chanmodes->execute($cn);
	my ($curmodes) = $get_chanmodes->fetchrow_array;
	my $ml = get_modelock($chan);

	# This method may exceed the 12-mode limit
	# But it seems to succeed anyway, even with more than 12.
	net::setmode($agent, $cn, modes::merge(modes::invert($curmodes), $ml, 1), '* *');
	do_modelock($chan);
}

sub cs_clear_ops($$;$) {
	my ($user, $chan, $reason) = @_;
	my $cn = $chan->{CHAN};
	my $src = get_user_nick($user);

	cs_clear_pre($user, $chan) or return;

	my $rlength = length($reason);
	if($rlength >= 350) {
		notice($user, 'Clear reason is too long by '. $rlength-350 .' character(s). Maximum length is 350 characters.');
		return;
	}

	clear_ops($chan);

	net::notice(agent($chan), $cn, "CLEAR OPS by \002$src\002".($reason?" reason: $reason":''));
}

sub cs_clear_bans($$;$$) {
	my ($user, $chan, $type, $reason) = @_;
	my $cn = $chan->{CHAN};
	my $src = get_user_nick($user);
	$type = 0 unless defined $type;

	cs_clear_pre($user, $chan) or return;

	my $rlength = length($reason);
	if($rlength >= 350) {
		notice($user, 'Clear reason is too long by '. $rlength-350 .' character(s). Maximum length is 350 characters.');
		return;
	}

	clear_bans($chan, $type);

	net::notice(agent($chan), $cn, "CLEAR BANS by \002$src\002".($reason?" reason: $reason":''));
}

sub cs_welcome_pre($$) {
	my ($user, $chan) = @_;

	return can_do($chan, 'WELCOME', undef, $user);
}

sub cs_welcome_add($$$) {
	my ($user, $chan, $msg) = @_;
	my $src = get_best_acc($user, $chan, 1);
	my $cn = $chan->{CHAN};

	cs_welcome_pre($user, $chan) or return;

	my $mlength = length($msg);
	if($mlength >= 350) {
		notice($user, 'Welcome Message is too long by '. $mlength-350 .' character(s). Maximum length is 350 characters.');
		return;
	}

	$count_welcome->execute($cn);
	my $count = $count_welcome->fetchrow_array;
	if ($count >= 5) {
		notice($user, 'There is a maximum of five (5) Channel Welcome Messages.');
		return;
	}

	$add_welcome->execute($cn, ++$count, $src, $msg);

	notice($user, "Welcome message number $count for \002$cn\002 set to:", "  $msg");
}

sub cs_welcome_list($$) {
	my ($user, $chan) = @_;
	my $cn = $chan->{CHAN};

	cs_welcome_pre($user, $chan) or return;

	$list_welcome->execute($cn);
	
	my @data;
	
	while(my ($id, $time, $adder, $msg) = $list_welcome->fetchrow_array) {
		push @data, ["$id)", $adder, scalar(misc::gmtime2($time)), '-', (length($msg) > 20 ? substr($msg, 0, 17) . '...' : $msg)];
	}
	$list_welcome->finish();

	notice($user,
		"Welcome message list for \002$cn\002:",
		fmt::columnar(@data)
	);
}

sub cs_welcome_del($$$) {
	my ($user, $chan, $id) = @_;
	my $cn = $chan->{CHAN};

	cs_welcome_pre($user, $chan) or return;

	if ($del_welcome->execute($cn, $id) == 1) {
		notice($user, "Welcome Message \002$id\002 deleted from \002$cn\002");
		$consolidate_welcome->execute($cn, $id);
	}
	else {
		notice($user,
			"Welcome Message number $id for \002$cn\002 does not exist.");
	}
}

sub cs_alist($$;$) {
        my ($user, $chan, $mask) = @_;
	my $cn = $chan->{CHAN};

	chk_registered($user, $chan) or return;

        my $slevel = get_best_acc($user, $chan);

	can_do($chan, 'ACCLIST', $slevel, $user) or return;

	my @reply;

	if($mask) {
		my ($mnick, $mident, $mhost) = misc::glob2sql(misc::parse_mask($mask));
		$mnick = '%' if($mnick eq '');
		$mident = '%' if($mident eq '');
		$mhost = '%' if($mhost eq '');

		$get_acc_list2_mask->execute($mnick, $cn, $mnick, $mident, $mhost);
		while(my ($n, $a, $l, $t, $lu, $id, $vh) = $get_acc_list2_mask->fetchrow_array) {
			push @reply, "*) $n ($id\@$vh) Rank: ".$levels[$l] . ($a ? ' Added by: '.$a : '');
			push @reply, '      '.($t ? 'Date/time added: '. scalar(misc::gmtime2($t)).' ' : '').
				($lu ? 'Last used '.misc::time_ago($lu).' ago' : '') if ($t or $lu);
		}
		$get_acc_list2_mask->finish();
	} else {
		$get_acc_list2->execute($cn);
		while(my ($n, $a, $l, $t, $lu, $id, $vh) = $get_acc_list2->fetchrow_array) {
			push @reply, "*) $n ($id\@$vh) Rank: ".$levels[$l] . ($a ? ' Added by: '.$a : '');
			push @reply, '      '.($t ? 'Date/time added: '. scalar(misc::gmtime2($t)).' ' : '').
				($lu ? 'Last used '.misc::time_ago($lu).' ago' : '') if ($t or $lu);
		}
		$get_acc_list2->finish();
	}

	notice($user, "Access list for \002$cn\002:", @reply);

	return;
}

sub cs_unban($$$) {
	my ($user, $chan, $parm) = @_;
	my $cn = $chan->{CHAN};

	my $self = 1 if (lc($parm) eq lc(get_user_nick($user)));
	if ($parm eq '*') {
		cs_clear_bans($user, $chan);
		return;
	}
	else {
		can_do($chan, ($self ? 'UnbanSelf' : 'UNBAN'), undef, $user) or return;
	}

	my $tuser = ($self ? $user : { NICK => $parm });
	unless(get_user_id($tuser)) {
		notice($user, "No such user: \002$parm\002");
		return;
	}
	# net::unban_nick disabled due to 3.2.5 bug (#2981) regarding cloakhosts
	# and now our code can handle cloakhosts, tho not extended bans.
	#if (defined(&net::unban_nick)) {
	#	net::unban_nick(agent($chan), $cn, get_user_nick($tuser));
	#} else {
		unban_user($chan, $tuser);
	#}
	notice($user, "All bans affecting ".($self ? 'you' : $parm)." on \002$cn\002 have been removed.");
}


sub cs_updown($$@) {
	my ($user, $cmd, @chans) = @_;
	return cs_updown2($user, $cmd, { CHAN => shift @chans }, @chans)
		if (defined($chans[1]) and $chans[1] !~ "^\#" and $chans[0] =~ "^\#");
	
	@chans = get_all_chans($user) 
		unless (@chans);

	if (uc($cmd) eq 'UP') {
		foreach my $cn (@chans) {
			next unless ($cn =~ /^\#/);
			my $chan = { CHAN => $cn };
			chanserv::set_modes($user, $chan, chanserv::get_best_acc($user, $chan));
		}
	}
	elsif (uc($cmd) eq 'DOWN') {
		foreach my $cn (@chans) {
			next unless ($cn =~ /^\#/);
			chanserv::unset_modes($user, { CHAN => $cn });
		}
	}
}

sub cs_updown2($$$@) {
	my ($user, $cmd, $chan, @targets) = @_;
	my $agent = $user->{AGENT} or $csnick;
	my $cn = $chan->{CHAN};

	return unless chk_registered($user, $chan);
	if (chk_flag($chan, F_FREEZE())) {
		notice($user, "\002$cn\002 is frozen and access suspended.");
		return;
	}

	my $acc = get_best_acc($user, $chan);
	return unless(can_do($chan, 'UPDOWN', $acc, $user));

	my $updown = ((uc($cmd) eq 'UP') ? 1 : 0);

	my ($override, $check_override);
	my (@list, $count);
	foreach my $target (@targets) {

		my $tuser = { NICK => $target };

		unless(is_in_chan($tuser, $chan)) {
			notice($user, "\002$target\002 is not in \002$cn\002.");
			next;
		}

		if($updown) {
			push @list, $target;
			chanserv::set_modes($tuser, $chan, chanserv::get_best_acc($tuser, $chan));
		}
		else {
			my $top = get_op($tuser, $chan);
			unless($top) {
				notice($user, "\002$target\002 is already deopped in \002$cn\002.");
				next;
			}

			if(!$override and get_best_acc($tuser, $chan) > $acc) {
				unless($check_override) {
					$override = adminserv::can_do($user, 'SUPER');
					$check_override = 1;
				}
				if($check_override and !$override) {
					notice($user, "\002$target\002 outranks you in \002$cn\002.");
					next;
				}
			}
			push @list, $target;
			chanserv::unset_modes($tuser, { CHAN => $cn });
		}
		$count++;
	}

	my $src = get_user_nick($user);
	net::notice(agent($chan), '%'.$cn, "$src used $cmd ".join(' ', @list))
		if (lc $user->{AGENT} eq lc $csnick) and chk_flag($chan, F_VERBOSE);
}

sub cs_getkey($$) {
	my ($user, $chan) = @_;
	my $cn = $chan->{CHAN};

	can_do($chan, 'GETKEY', undef, $user) or return;

	$get_chanmodes->execute($cn);
	my $modes = $get_chanmodes->fetchrow_array; $get_chanmodes->finish();

	if(my $key = modes::get_key($modes)) {
		notice($user, "Channel key for \002$cn\002: $key");
	}
	else {
		notice($user, "\002$cn\002 has no channel key.");
	}
}

sub cs_auth($$$@) {
	my ($user, $chan, $cmd, @args) = @_;
	my $cn = $chan->{CHAN};
	$cmd = lc $cmd;

	return unless chk_registered($user, $chan);
	return unless can_do($chan, 'AccChange', $user);
	my $userlevel = get_best_acc($user, $chan);
	if($cmd eq 'list') {
		my @reply = ['Nick', 'Level', 'Adder', 'Time'];
		$list_auth_chan->execute($cn);
		while(my ($nick, $data) = $list_auth_chan->fetchrow_array()) {
			my ($adder, $old, $level, $time) = split(/:/, $data);
			push @reply, [$nick, $levels[$level], $adder, misc::gmtime2($time)];
		}
		if ($list_auth_chan->rows()) {
			notice($user, fmt::columnar(@reply));
		}
		else {
			notice($user, "There are no pending authorizations for \002$cn\002");
		}
		$list_auth_chan->finish();
	}
	elsif($cmd eq 'remove' or $cmd eq 'delete' or $cmd eq 'del') {
	my ($nick, $adder, $old, $level, $time);
	my $parm = shift @args;
		if(misc::isint($parm) and ($nick, $adder, $old, $level, $time) = get_auth_num($cn, $parm))
		{
		}
		elsif (($adder, $old, $level, $time) = get_auth_nick($cn, $parm))
		{
			$nick = $parm;
		}
		unless ($nick) {
		# This should normally be an 'else' as the elsif above should prove false
		# For some reason, it doesn't work. the unless ($nick) fixes it.
		# It only doesn't work for numbered entries
			notice($user, "There is no entry for \002$parm\002 in \002$cn\002's AUTH list");
			return;
		}
		$nickserv::del_auth->execute($nick, $cn); $nickserv::del_auth->finish();
		my $log_str = "deleted AUTH entry $cn $nick $levels[$level]";
		my $src = get_user_nick($user);
		notice($user, "You have $log_str");
		net::notice(agent($chan), '%'.$cn, "has \002$src\002 has $log_str")
			if chk_flag($chan, F_VERBOSE);
		services::ulog($chanserv::csnick, log::INFO(), "has $log_str", $user, $chan);
	}
	else {
		notice($user, "Unknown AUTH command \002$cmd\002");
	}
}

### MISCELLANEA ###

sub do_welcome($$) {
	my ($user, $chan) = @_;
	my $cn = $chan->{CHAN};
	
	$get_welcomes->execute($cn);
	if($get_welcomes->rows) {
		my @welcomes;
		while(my ($msg) = $get_welcomes->fetchrow_array) {
			push @welcomes, (chk_flag($chan, F_WELCOMEINCHAN) ? '' : "[$cn] " ).$msg;
		}
		if(chk_flag($chan, F_WELCOMEINCHAN)) {
			net::privmsg(agent($chan), $cn, @welcomes);
		} else {
			notice($user, @welcomes);
		}
	}
	$get_welcomes->finish();
}

sub do_greet($$) {
	my ($user, $chan) = @_;
	my $cn = $chan->{CHAN};

	if(can_do($chan, 'GREET', undef, $user)) {
		my $src = get_user_nick($user);
		$nickserv::get_greet->execute(get_user_id($user));
		my ($greet) = $nickserv::get_greet->fetchrow_array();
		$nickserv::get_greet->finish();
		net::privmsg(agent($chan), $cn, "[\002$src\002] $greet") if $greet;
	}
}

sub chk_registered($$) {
	my ($user, $chan) = @_;

	unless(is_registered($chan)) {
		my $cn = $chan->{CHAN};
		
		notice($user, "The channel \002$cn\002 is not registered.");
		return 0;
	}

	return 1;
}

sub make_banmask($$) {
	my ($chan, $tuser) = @_;
	my $nick = get_user_nick($tuser);

	my ($ident, $vhost) = nickserv::get_vhost($tuser);
	my ($nick, $ident, $vhost) = misc::make_hostmask(get_bantype($chan), $nick, $ident, $vhost);
	return "$nick!$ident\@$vhost";
}

sub kickban($$$$) {
	my ($chan, $user, $mask, $reason) = @_;
	my $cn = $chan->{CHAN};
	my $nick = get_user_nick($user);

	return 0 if adminserv::is_service($user);

	my $agent = agent($chan);

	unless($mask) {
		$mask = make_banmask($chan, $user);
	}

	enforcer_join($chan) if (get_user_count($chan) <= 1);
	net::setmode($agent, $cn, '+b', $mask);
	net::flushmodes();
	net::kick($agent, $cn, $nick, $reason);
	return 1;
}

sub kickban_multi($$$) {
	my ($chan, $users, $reason) = @_;
	my $cn = $chan->{CHAN};
	my $agent = agent($chan);
	
	enforcer_join($chan);
	net::setmode($agent, $cn, '+b', '*!*@*');
	net::flushmodes();

	foreach my $user (@$users) {
		next if adminserv::is_ircop($user) or adminserv::is_svsop($user, adminserv::S_HELP());
		net::kick($agent, $cn, get_user_nick($user), $reason);
	}
}

sub clear_users($$)  {
	my ($chan, $reason) = @_;
	my $cn = $chan->{CHAN};
	my $agent = agent($chan);
	my $i;
	
	enforcer_join($chan);
	net::setmode($agent, $cn, '+b', '*!*@*');
	net::flushmodes();
	$get_chan_users->execute($cn);
	while(my ($nick, $uid) = $get_chan_users->fetchrow_array) {
		my $user = { NICK => $nick, ID => $uid };
		net::kick($agent, $cn, $nick, $reason)
			unless adminserv::is_ircop($user) or adminserv::is_svsop($user, adminserv::S_HELP());
		$i++;
	}

	return $i;
}

sub kickmask($$$$)  {
	my ($chan, $mask, $reason, $ban) = @_;
	my $cn = $chan->{CHAN};
	my $agent = agent($chan);

	my ($nick, $ident, $host) = misc::glob2sql(misc::parse_mask($mask));
	$nick = '%' if ($nick eq '');
	$ident = '%' if ($ident eq '');
	$host = '%' if ($host eq '');
	
	if ($ban) {
		my $banmask = $nick.'!'.$ident.'@'.$host;
		$banmask =~ tr/%_/*?/;
		net::setmode($agent, $cn, '+b', $banmask);
		net::flushmodes();
	}

	my $i;
	$get_chan_users_mask->execute($cn, $nick, $ident, $host, $host, $host);
	while(my ($nick, $uid) = $get_chan_users_mask->fetchrow_array) {
		my $user = { NICK => $nick, ID => $uid };
		net::kick($agent, $cn, $nick, $reason)
			unless adminserv::is_service($user);
		$i++;
	}
	$get_chan_users_mask->finish();

	return $i;
}

sub kickmask_noacc($$$$)  {
	my ($chan, $mask, $reason, $ban) = @_;
	my $cn = $chan->{CHAN};
	my $agent = agent($chan);

	my ($nick, $ident, $host) = misc::glob2sql(misc::parse_mask($mask));
	$nick = '%' if ($nick eq '');
	$ident = '%' if ($ident eq '');
	$host = '%' if ($host eq '');
	
	if ($ban) {
		my $banmask = $nick.'!'.$ident.'@'.$host;
		$banmask =~ tr/%_/*?/;
		net::setmode($agent, $cn, '+b', $banmask);
		net::flushmodes();
	}

	my $i;
	$get_chan_users_mask_noacc->execute($cn, $nick, $ident, $host, $host, $host);
	while(my ($nick, $uid) = $get_chan_users_mask_noacc->fetchrow_array) {
		my $user = { NICK => $nick, ID => $uid };
		net::kick($agent, $cn, $nick, $reason)
			unless adminserv::is_service($user);
		$i++;
	}
	$get_chan_users_mask_noacc->finish();

	return $i;
}

sub clear_ops($) {
	my ($chan) = @_;
	my $cn = $chan->{CHAN};
	my @modelist;
	my $agent = agent($chan);

	$get_chan_users->execute($cn);
	while(my ($nick, $uid) = $get_chan_users->fetchrow_array) {
		my $user = { NICK => $nick, ID => $uid };
		my $opmodes = get_op($user, $chan);
		for(my $i; $i < 5; $i++) {
			if($opmodes & 2**$i) {
				push @modelist, ['-'.$opmodes[$i], $nick];
			}
		}
	}

	net::setmode2($agent, $cn, @modelist);
}

sub clear_bans($;$) {
	my ($chan, $type) = @_;
	my $cn = $chan->{CHAN};
	my @args = ();
	my $agent = agent($chan);
	$type = 0 unless defined $type;
	my $mode = ($type == 128 ? 'e' : 'b');
	
	my @banlist = ();
	$get_all_bans->execute($cn, $type);
	while(my ($mask) = $get_all_bans->fetchrow_array) {
		$mask =~ tr/\%\_/\*\?/;
		push @banlist, $mask;
	}

	net::ban_list($agent, $cn, -1, $mode, @banlist);
	net::flushmodes();
}

sub unban_user($$) {
	my ($chan, $tuser) = @_;
	my $cn = $chan->{CHAN};
	my $tuid;
	unless($tuid = get_user_id($tuser)) {
		return 0;
	}

	my (@bans);
	# We don't handle extended bans. Yet.
	$find_bans_chan_user->execute($cn, $tuid, 0);
	while (my ($mask) = $find_bans_chan_user->fetchrow_array) {
		$mask =~ tr/\%\_/\*\?/;
		push @bans, $mask;
	}
	$find_bans_chan_user->finish();

	net::ban_list(agent($chan), $cn, -1, 'b', @bans) if scalar(@bans);
	$delete_bans_chan_user->execute($cn, $tuid, 0); $delete_bans_chan_user->finish();
}

sub chan_kill($$;$)  {
	my ($chan, $reason, $users) = @_;
	my $cn = $chan->{CHAN};
	my $agent = agent($chan);
	my $i;
	
	enforcer_join($chan);
	if ($users) {
		foreach my $user (@$users) {
			$user->{ID} = $user->{__ID} if defined($user->{__ID}); # user_join_multi does this.
			nickserv::kline_user($user, $services::conf{chankilltime}, $reason)
				unless adminserv::is_ircop($user) or adminserv::is_svsop($user, adminserv::S_HELP());
			$i++;
		}
	}
	else {
		$get_chan_users->execute($cn);
		while(my ($nick, $uid) = $get_chan_users->fetchrow_array) {
			my $user = { NICK => $nick, ID => $uid, AGENT => $agent };
			nickserv::kline_user($user, $services::conf{chankilltime}, $reason)
				unless adminserv::is_ircop($user) or adminserv::is_svsop($user, adminserv::S_HELP());
			$i++;
		}
	}

	return $i;
}

sub do_nick_akick($$;$) {
	my ($tuser, $chan, $root) = @_;
	my $cn = $chan->{CHAN};
	unless(defined($root)) {
		(undef, $root) = get_best_acc($tuser, $chan, 2);
	}
	$get_nick_akick->execute($cn, $root);
	my ($reason) = $get_nick_akick->fetchrow_array(); $get_nick_akick->finish();
	kickban($chan, $tuser, undef, "User has been banned from ".$cn.($reason?": $reason":''));
}

sub do_status($$) {
	my ($user, $chan) = @_;

	return 0 if chk_flag($chan, (F_CLOSE | F_DRONE));
	
	my $uid = get_user_id($user);
	my $nick = get_user_nick($user);
	my $cn = $chan->{CHAN};

	my ($acc, $root) = get_best_acc($user, $chan, 2);
	if ($acc == -1) {
		do_nick_akick($user, $chan, $root);
		return 0;
	}
	unless(can_do($chan, 'JOIN', $acc, $user)) {
		kickban($chan, $user, undef, 'This is a private channel.');
		return 0;
	}
	
	unless($acc or adminserv::is_svsop($user, adminserv::S_HELP()) ) {
		$get_akick->execute($uid, $cn);
		if(my @akick = $get_akick->fetchrow_array) {
			akickban($cn, @akick);
			return 0;
		}
	}
	
	set_modes($user, $chan, $acc, chk_flag($chan, F_SPLITOPS, 0))
		if is_registered($chan)
		and not is_neverop($root)
		and not chk_flag($chan, (F_CLOSE | F_DRONE));
	
	return 1;
}

sub akick_alluser($) {
	my ($user) = @_;
	my $uid = get_user_id($user);

	$get_akick_alluser->execute($uid);
	while(my @akick = $get_akick_alluser->fetchrow_array) {
		akickban(@akick);
	}
}

sub akick_allchan($) {
	my ($chan) = @_;
	my $cn = $chan->{CHAN};

	$get_akick_allchan->execute($cn);
	while(my @akick = $get_akick_allchan->fetchrow_array) {
		akickban($cn, @akick);
	}
}

sub akickban(@) {
	my ($cn, $knick, $bnick, $ident, $host, $reason, $bident) = @_;

	my $target = { NICK => $knick };
	my $chan = { CHAN => $cn };
	return 0 if adminserv::is_svsop($target, adminserv::S_HELP());

	if($bident) {
		($bnick, $ident, $host) = misc::make_hostmask(get_bantype($chan), $knick, $bident, $host);
	} else {
		$bnick =~ tr/\%\_/\*\?/;
		$ident =~ tr/\%\_/\*\?/;
		$host =~ tr/\%\_/\*\?/;
	}

	return kickban($chan, $target, "$bnick!$ident\@$host", "User has been banned from ".$cn.($reason?": $reason":''));
}

sub notice_all_nicks($$$) {
	my ($user, $nick, $msg) = @_;
	my $src = get_user_nick($user);

	notice($user, $msg);
	$nickserv::get_using_nicks->execute($nick);
	while(my ($u) = $nickserv::get_using_nicks->fetchrow_array) {
		net::notice($csnick, $u, $msg) unless $src =~ /^\Q$u\E$/i;
	}
}

sub xop_byname($) {
	my ($name) = @_;
	my $level;

	if($name =~ /^uop$/i) { $level=1; }
	elsif($name =~ /^vop$/i) { $level=2; }
	elsif($name =~ /^hop$/i) { $level=3; }
	elsif($name =~ /^aop$/i) { $level=4; }
	elsif($name =~ /^sop$/i) { $level=5; }
	elsif($name =~ /^co?f(ounder)?$/i) { $level=6; }
	elsif($name =~ /^founder$/i) { $level=7; }
	elsif($name =~ /^(any|all|user)/i) { $level=0; }
	elsif($name =~ /^akick$/i) { $level=-1; }
	elsif($name =~ /^(none|disabled?|nobody)$/i) { $level=8; }

	return $level;
}

sub expire {
	$get_expired->execute(time() - (86400 * $services::conf{'chanexpire'}));
	while(my ($cn, $founder) = $get_expired->fetchrow_array) {
		drop({ CHAN => $cn });
		log::wlog($csnick, log::INFO(), "\002$cn\002 has expired.  Founder: $founder");
	}
}

sub enforcer_join($) {
	my ($chan) = @_;
	my $cn = $chan->{CHAN};
	my $bot = agent($chan);

	return if $enforcers{lc $cn};
	$enforcers{lc $cn} = lc $bot;

	botserv::bot_join($chan);
	
	add_timer("CSEnforce $bot $cn", 60, __PACKAGE__, 'chanserv::enforcer_part');
}

sub enforcer_part($) {
	my ($cookie) = @_;
	my ($junk, $bot, $cn) = split(/ /, $cookie);

	return unless $enforcers{lc $cn};
	undef($enforcers{lc $cn});
	
	botserv::bot_part_if_needed($bot, {CHAN => $cn}, 'Enforcer Leaving');
}

### DATABASE UTILITY FUNCTIONS ###

sub get_user_count($) {
	my ($chan) = @_;
	my $cn = $chan->{CHAN};
	
	$get_user_count->execute($cn);
	
	return $get_user_count->fetchrow_array;
}

sub get_lock($) {
	my ($chan) = @_;

	$chan = lc $chan;

	$chanuser_table++;

	if($cur_lock) {
		if($cur_lock ne $chan) {
			really_release_lock($chan);
			$chanuser_table--;
			die("Tried to get two locks at the same time: $cur_lock, $chan")
		}
		$cnt_lock++;
	} else {
		$cur_lock = $chan;
		$get_lock->execute($services::conf{'mysql-db'}.".chan.$chan");
		$get_lock->finish;
	}	
}

sub release_lock($) {
	my ($chan) = @_;

	$chan = lc $chan;

	$chanuser_table--;

	if($cur_lock and $cur_lock ne $chan) {
		really_release_lock($cur_lock);
		
		die("Tried to release the wrong lock");
	}

	if($cnt_lock) {
		$cnt_lock--;
	} else {
		really_release_lock($chan);
	}
}

sub really_release_lock($) {
	my ($chan) = @_;

	$cnt_lock = 0;
	$release_lock->execute($services::conf{'mysql-db'}.".chan.$chan");
	$release_lock->finish;
	undef $cur_lock;
}

#sub is_free_lock($) {
#	$is_free_lock->execute($_[0]);
#	return $is_free_lock->fetchrow_array;
#}

sub get_modelock($) {
	my ($chan) = @_;
	my $cn;
	if(ref($chan)) {
		$cn = $chan->{CHAN}
	} else {
		$cn = $chan;
	}
							
	$get_modelock->execute($cn);
	my ($ml) = $get_modelock->fetchrow_array;
	$get_modelock->finish();
	return $ml;
}

sub do_modelock($;$) {
	my ($chan, $modes) = @_;
	my $cn = $chan->{CHAN};

	my $seq = $ircline;

	$get_modelock_lock->execute; $get_modelock_lock->finish;
	
	$get_chanmodes->execute($cn);
	my ($omodes) = $get_chanmodes->fetchrow_array;
	my $ml = get_modelock($chan);

	$ml = do_modelock_fast($cn, $modes, $omodes, $ml);

	$unlock_tables->execute; $unlock_tables->finish;

	net::setmode(agent($chan), $cn, $ml) if($ml);
}

sub do_modelock_fast($$$$) {
	my ($cn, $modes, $omodes, $ml) = @_;
	my $nmodes = modes::add($omodes, $modes, 1);
	$ml = modes::diff($nmodes, $ml, 1);
	$set_chanmodes->execute(modes::add($nmodes, $ml, 1), $cn);
	
	return $ml;
}

sub update_modes($$) {
	my ($cn, $modes) = @_;

	$get_update_modes_lock->execute; $get_update_modes_lock->finish;
	$get_chanmodes->execute($cn);
	my ($omodes) = $get_chanmodes->fetchrow_array;

	$set_chanmodes->execute(modes::add($omodes, $modes, 1), $cn);
	$unlock_tables->execute; $unlock_tables->finish;
}

sub is_level($) {
	my ($perm) = @_;

	$is_level->execute($perm);
	
	return $is_level->fetchrow_array;
}

sub set_flag($$$) {
	my ($chan, $flag, $sign) = @_;
	my $cn = $chan->{CHAN};

	if($sign) {
		$set_flag->execute($flag, $flag, $cn);
	} else {
		$unset_flag->execute($flag, $flag, $cn);
	}
}

sub chk_flag($$;$) {
	my ($chan, $flag, $sign) = @_;
	my $cn = $chan->{CHAN};
	$sign = 1 unless defined($sign);

	my $flags;
	unless (exists($chan->{FLAGS})) {
		$get_flags->execute($cn);
		($flags) = $get_flags->fetchrow_array;
	}
	else {
		$flags = $chan->{FLAGS};
	}

	return ($sign ? ($flags & $flag) : !($flags & $flag));
}

sub is_neverop($) {
	return nickserv::chk_flag($_[0], nickserv::F_NEVEROP(), 1);
}

sub is_in_chan($$) {
	my ($user, $chan) = @_;
	my $cn = $chan->{CHAN};
	my $uid = get_user_id($user);

	$is_in_chan->execute($uid, $cn);
	if($is_in_chan->fetchrow_array) {
		return 1;
	}

	return 0;
}

sub is_registered($) {
	my ($chan) = @_;
	my $cn = $chan->{CHAN};
	
	$is_registered->execute($cn);
	if($is_registered->fetchrow_array) {
		return 1;
	} else {
		return 0;
	}
}

sub get_all_chans($) {
	my ($user) = @_;
	my $uid = get_user_id($user);
	my @chans;
	
	$get_all_chans->execute($uid, $ircline, $ircline+1000);
	while(my ($chan) = $get_all_chans->fetchrow_array) {
		push @chans, $chan;
	}

	return (@chans);
}

sub get_op($$) {
	my ($user, $chan) = @_;
	my $cn = $chan->{CHAN};
	my $uid = get_user_id($user);

	$get_op->execute($uid, $cn);
	my ($op) = $get_op->fetchrow_array;

	return $op;
}

sub get_best_acc($$;$) {
	my ($user, $chan, $retnick) = @_;
	my $uid = get_user_id($user);
	my $cn = $chan->{CHAN};

	return undef
		if chk_flag($chan, (F_DRONE | F_CLOSE | F_FREEZE), 1);

	$get_best_acc->execute($uid, $cn);
	my ($bnick, $best) = $get_best_acc->fetchrow_array;

	if($retnick == 2) {
		return ($best, $bnick);
	} elsif($retnick == 1) {
		return $bnick;
	} else {
		return $best;
	}
}

sub get_acc($$) {
	my ($nick, $chan) = @_;
	my $cn = $chan->{CHAN};

	return undef
		if chk_flag($chan, (F_DRONE | F_CLOSE | F_FREEZE), 1);

	$get_acc->execute($cn, $nick);
	my ($acc) = $get_acc->fetchrow_array;
	
	return $acc;
}

sub set_acc($$$$) {
	my ($nick, $user, $chan, $level) = @_;
	my $cn = $chan->{CHAN};
	my $adder = get_best_acc($user, $chan, 1) if $user;

	$set_acc1->execute($cn, $level, $nick);
	$set_acc2->execute($level, $adder, $cn, $nick);

	set_modes_allnick($nick, $chan, $level) if $level < 0 or !is_neverop($nick);
}

sub del_acc($$) {
	my ($nick, $chan) = @_;
	my $cn = $chan->{CHAN};

	$del_acc->execute($cn, $nick);

	$nickserv::get_using_nicks->execute($nick);
	while(my ($nick, $uid) = $nickserv::get_using_nicks->fetchrow_array) {
		my $user = { NICK => $nick, ID => $uid };
		set_modes($user, $chan, 0, 1) if is_in_chan($user, $chan);
	}
	$nickserv::get_using_nicks->finish();
}

sub get_auth_nick($$) {
	my ($cn, $nick) = @_;

	$get_auth_nick->execute($cn, $nick);
	my ($data) = $get_auth_nick->fetchrow_array();
	$get_auth_nick->finish();

	return split(/:/, $data);
}
sub get_auth_num($$) {
	my ($cn, $num) = @_;

	$get_auth_num->execute($cn, $num - 1);
	my ($nick, $data) = $get_auth_num->fetchrow_array();
	$get_auth_num->finish();

	return ($nick, split(/:/, $data));
}
sub find_auth($$) {
	my ($cn, $nick) = @_;

	$find_auth->execute($cn, $nick);
	my ($ret) = $find_auth->fetchrow_array();
	$find_auth->finish();

	return $ret;
}

# Only call this if you've checked for NEVEROP already.
sub set_modes_allchan($) {
	my ($user) = @_;
	my $uid = get_user_id($user);

	$get_all_chans->execute($uid, $ircline, $ircline+1000);
	while(my ($cn) = $get_all_chans->fetchrow_array) {
		my $chan = { CHAN => $cn };
		my $acc = get_best_acc($user, $chan);
		if($acc > 0) {
			set_modes($user, $chan, $acc);
		} elsif($acc < 0) {
			do_nick_akick($user, $chan);
		}
	}
}

# Only call this if you've checked for NEVEROP already.
sub set_modes_allnick($$$) {
	my ($nick, $chan, $level) = @_;
	my $cn = $chan->{CHAN};
	
	$get_using_nick_chans->execute($nick, $cn);
	while(my ($n) = $get_using_nick_chans->fetchrow_array) {
		my $user = { NICK => $n };
		my $l = get_best_acc($user, $chan);
		if($l > 0) {
			set_modes($user, $chan, $level, 1) if($level == $l);
		} elsif($l < 0) {
			do_nick_akick($user, $chan);
		}
	}
}

# If channel has OPGUARD, $doneg is true.
sub set_modes($$$;$) {
	my ($user, $chan, $acc, $doneg) = @_;
	my $cn = $chan->{CHAN};

	
	if ($acc < 0) {
	# Do akick stuff here.
	}
	
	my $dst = $ops[$acc];
	my $cur = get_op($user, $chan);
	my ($pos, $neg);
	
	if (chk_flag($chan, F_FREEZE)) {
		set_mode_mask($user, $chan, $cur, undef);
		return;
	}
	if (($acc == 0) and chk_flag($chan, F_AUTOVOICE)) {
		set_mode_mask($user, $chan, $cur, 1);
		return;
	}

	$pos = $dst ^ ($dst & $cur);
	$neg = ($dst ^ $cur) & $cur if $doneg;

	if($pos or $neg) {
		set_mode_mask($user, $chan, $neg, $pos);
	}

	if($pos) {
		$set_lastop->execute($cn);
		$set_lastused->execute($cn, get_user_id($user));
	}
}

sub unset_modes($$) {
	my ($user, $chan) = @_;

	my $mask = get_op($user, $chan);

	set_mode_mask($user, $chan, $mask, 0);
}

sub set_mode_mask($$$$) {
	my ($user, $chan, @masks) = @_;
	my $nick = get_user_nick($user);
	my $cn = $chan->{CHAN};
	my (@args, $out);

	for(my $sign; $sign < 2; $sign++) {
		next if($masks[$sign] == 0);

		$out .= '-' if $sign == 0;
		$out .= '+' if $sign == 1;

		for(my $i; $i < 5; $i++) {
			my @l = ('v', 'h', 'o', 'a', 'q');

			if($masks[$sign] & 2**$i) {
				$out .= $l[$i];
				push @args, $nick;
			}
		}
	}

	if(@args) {
		net::setmode(agent($chan), $cn, $out, join(' ', @args));
	}
}

sub get_level($$) {
	my ($chan, $perm) = @_;
	my $cn = $chan->{CHAN};

	$get_level->execute($cn, $perm);
	my ($level, $isnotnull) = $get_level->fetchrow_array;
	$get_level->finish();

	if (wantarray()) {
		return ($level, $isnotnull);
	}
	else {
		return $level;
	}
}

sub check_override($$) {
	my ($user, $perm) = @_;

	foreach my $o (@override) {
		if($o->[1]{uc $perm} and my $nick = adminserv::can_do($user, $o->[0])) {
			return (wantarray ? ($nick, 1) : $nick);
		}
	}
}

sub can_do($$$;$$) {
	my ($chan, $perm, $acc, $user, $noreply) = @_;
	my $nick;
	my $cn = $chan->{CHAN};
	
	if ($user and adminserv::is_svsop($user, adminserv::S_HELP()) and 
		my ($nick, $override) = check_override($user, $perm)) 
	{
		$set_lastused->execute($cn, get_user_id($user));
		return (wantarray ? ($nick, $override) : $nick) if $override;
	}

	my $level = get_level($chan, $perm);

	unless(defined($acc)) {
		($acc, $nick) = get_best_acc($user, $chan, 2);
	}
	$nick = 1 unless $nick;

	if($acc >= $level and !chk_flag($chan, (F_CLOSE | F_FREEZE | F_DRONE))) {
		$set_lastused->execute($cn, get_user_id($user)) if $user;
		return (wantarray ? ($nick, 0) : $nick);
	}

	if(chk_flag($chan, F_FREEZE) and ($perm eq 'JOIN')) {
		return (wantarray ? ($nick, 0) : $nick);
	}

	if($user and !$noreply) {
		if (chk_flag($chan, (F_CLOSE | F_DRONE))) {
			notice($user, "\002$cn\002 is closed and cannot be used".
				((uc $perm eq 'INFO') ? ': '.get_close($chan) : '.'));
		}
		elsif(chk_flag($chan, F_FREEZE)) {
			notice($user, "\002$cn\002 is frozen and access suspended.");
		}
		else {
			notice($user, "$cn: $err_deny");
		}
	}

	return 0;
}

sub can_keep_op($$$$) {
# This is a nave implemenation using a loop.
# If we ever do a more flexible version that further restricts how
# LEVELS affect opguard, the loop will have to be unrolled.
# --
# Only call this if you've already checked opguard, as we do not check it here.
# -- 
# Remember, this isn't a permission check if someone is allowed to op someone [else],
# rather this checks if the person being opped is allowed to keep/have it.
	my ($user, $chan, $tuser, $opmode) = @_;
	return 1 if $opmode eq 'v'; # why remove a voice?
	my %permhash = (
		'q' => ['OWNER', 	4],
		'a' => ['ADMIN', 	3],
		'o' => ['OP', 		2],
		'h' => ['HALFOP', 	1],
		'v' => ['VOICE', 	0]
	);

	my $self = (lc(get_user_nick($user)) eq lc(get_user_nick($tuser)));

	#my ($level, $isnotnull) = get_level($chan, $permhash{$opmode}[1]);
	my $level = get_level($chan, $permhash{$opmode}[0]);

	foreach my $luser ($tuser, $user) {
	# We check target first, as there seems no reason that
	# someone who has access can't be opped by someone
	# who technically doesn't.
		return 1 if (adminserv::is_svsop($luser, adminserv::S_HELP()) and
			check_override($luser, $permhash{$opmode}[0]));

		my $acc = get_best_acc($luser, $chan);
		return 1 if ($self and ($permhash{opmode}[2] + 2) <= $acc);

		if($acc < $level) {
			return 0;
		}
	}

	return 1;
}

sub agent($) {
	my ($chan) = @_;

	return $chan->{AGENT} if($chan->{AGENT});
	
	unless(synced()) {
		return $csnick;
	}

	$botserv::get_chan_bot->execute($chan->{CHAN});
	my ($agent) = $botserv::get_chan_bot->fetchrow_array;

	$agent = $csnick unless $agent;

	return $chan->{AGENT} = $agent;
}

sub drop($) {
	my ($chan) = @_;
	my $cn = $chan->{CHAN};

	undef($enforcers{lc $cn});
	my $agent = agent($chan);
	agent_part($agent, $cn, 'Channel dropped') unless (lc($agent) eq lc($csnick));

	$drop_acc->execute($cn);
	$drop_lvl->execute($cn);
	$del_close->execute($cn);
	$drop_akick->execute($cn);
	$drop_welcome->execute($cn);
	$drop_chantext->execute($cn);
	$drop_nicktext->execute($cn); # Leftover channel auths
	$drop->execute($cn);
	net::setmode($csnick, $cn, '-r');
}

sub drop_nick_chans($) {
	my ($nick) = @_;

	$delete_successors->execute($nick);
	
	$get_nick_own_chans->execute($nick);
	while(my ($cn) = $get_nick_own_chans->fetchrow_array) {
		succeed_chan($cn, $nick);
	}
}

sub succeed_chan($$) {
	my ($cn, $nick) = @_;

	$get_successor->execute($cn);
	my ($suc) = $get_successor->fetchrow_array;

	if($suc) {
		$set_founder->execute($suc, $cn);
		set_acc($suc, undef, {CHAN => $cn}, FOUNDER);
		$del_successor->execute($cn);
	} else {
		drop({CHAN => $cn});
		log::wlog($csnick, log::INFO(), "\002$cn\002 has been dropped due to expiry/drop of \002$nick\002");
	}
}

sub get_close($) {
	my ($chan) = @_;
	my $cn = $chan->{CHAN};
	return undef unless chk_flag($chan, F_CLOSE | F_DRONE);

	$get_close->execute($cn);
	my ($reason, $opnick, $time) = $get_close->fetchrow_array();
	$get_close->finish();

	$reason = "[$opnick ".misc::gmtime2($time)."] - $reason";
	
	return (wantarray ? ($reason, $opnick, $time) : $reason);
}

sub get_users_nochans(;$) {
	my ($noid) = @_;
	my @users;

	if($noid) {
		$get_users_nochans_noid->execute();
		while (my ($usernick, $userid) = $get_users_nochans_noid->fetchrow_array()) {
			push @users, { NICK => $usernick, ID => $userid };
		}
		$get_users_nochans_noid->finish();
	}
	else {
		$get_users_nochans->execute();
		while (my ($usernick, $userid) = $get_users_nochans->fetchrow_array()) {
			push @users, { NICK => $usernick, ID => $userid };
		}
		$get_users_nochans->finish();
	}

	return @users;
}

sub get_bantype($) {
	my ($chan) = @_;
	my $cn = $chan->{CHAN};

	$get_bantype->execute($cn);
	my ($bantype) = $get_bantype->fetchrow_array();
	$get_bantype->finish();

	return $bantype;
}

### IRC EVENTS ###

sub user_join($$) {
# Due to special casing of '0' this wrapper should be used
# by anyone handling a JOIN (not SJOIN, it's a JOIN) event.
# This is an RFC1459 requirement.
	my ($nick, $cn) = @_;
	my $user = { NICK => $nick };
	my $chan = { CHAN => $cn };

	if ($cn == 0) {
	# This should be treated as a number
	# Just in case we ever got passed '000', not that Unreal does.
	# In C, you could check that chan[0] != '#' && chan[0] == '0'
		user_part_multi($user, [ get_all_chans($user) ], 'Left all channels');
	}
	else {
		user_join_multi($chan, [$user]);
	}
}

sub handle_sjoin($$$$$$$) {
	my ($server, $cn, $ts, $chmodes, $chmodeparms, $userarray, $banarray, $exceptarray) = @_;
	my $chan = { CHAN => $cn };

	if(synced()) {
		chan_mode($server, $cn, $chmodes, $chmodeparms) if $chmodes;
	} else {
		update_modes($cn, "$chmodes $chmodeparms") if $chmodes;
	}
	user_join_multi($chan, $userarray) if $userarray;

	foreach my $ban (@$banarray) {
		process_ban($cn, $ban, $server, 0, 1);
	}
	foreach my $except (@$exceptarray) {
		process_ban($cn, $except, $server, 128, 1);
	}
}

sub user_join_multi($$) {
	my ($chan, $users) = @_;
	my $cn = $chan->{CHAN};
	my $seq = $ircline;

	foreach my $user (@$users) {
		$user->{__ID} = get_user_id($user);
	}
	
	$get_joinpart_lock->execute; $get_joinpart_lock->finish;

	$chan_create->execute($seq, $cn);

	$get_user_count->execute($cn);
	my ($count) = $get_user_count->fetchrow_array;

	foreach my $user (@$users) {
		$joinpart1->execute($seq, $user->{__ID}, $cn);
		$joinpart2->execute($seq, 1, ($user->{__OP} or 0), $user->{__ID}, $cn, $seq, $seq+1000);
	}

	$unlock_tables->execute; $unlock_tables->finish;

	my $bot = agent($chan);
	foreach my $user (@$users) {
		$user->{AGENT} = $bot;
	}
	
	if(synced() and chk_flag($chan, (F_CLOSE | F_DRONE))) {
		my ($reason, $opnick, $time) = get_close($chan);
		my $cmsg = "$cn is closed: $reason";
		my $preenforce = $enforcers{lc $chan};
		
		if (chk_flag($chan, F_CLOSE)) {
			kickban_multi($chan, $users, $cmsg);
		}
		elsif (chk_flag($chan, F_DRONE)) {
			chan_kill($chan, $cmsg, $users);
		}

		unless($preenforce) {
			net::settopic($bot, $cn, $opnick, $time, $cmsg);

			my $ml = get_modelock($chan);
			net::setmode($bot, $cn, $ml) if($ml);
		}
	}

	if($count == 0 and initial_synced()) {
		unless (lc($bot) eq lc($csnick)) {
			unless(is_agent_in_chan($bot, $cn)) {
				botserv::bot_join($chan);
			}
		}
	}
	
	return unless synced() and not chk_flag($chan, (F_CLOSE | F_DRONE));

	my $n;
	foreach my $user (@$users) {
		if(do_status($user, $chan)) {
			$n++;
			$user->{__DO_WELCOME} = 1;
		}
	}

	if($count == 0 and $n) {
		my ($ml) = get_modelock($chan);
		net::setmode($bot, $cn, $ml) if($ml);
		
		$get_topic->execute($cn);
		my ($ntopic, $nsetter, $ntime) = $get_topic->fetchrow_array;
		net::settopic($bot, $cn, $nsetter, $ntime, $ntopic) if $ntopic;
	}

	net::flushmodes();

	if($n) {
		foreach my $user (@$users) {
			if ($user->{__DO_WELCOME} and nickserv::chk_user_flag($user, nickserv::UF_FINISHED())) {
				do_welcome($user, $chan);
				do_greet($user, $chan)
					if can_do($chan, 'GREET', undef, $user, 1);
			}
		}
	}
}

sub user_part($$$) {
	my ($nick, $cn, $reason) = @_;

	my $user = { NICK => $nick };

	user_part_multi($user, [ $cn ], $reason);
}

sub user_part_multi($$$) {
# user_join_multi takes a channel and multiple users
# user_part_multi takes a user and multiple channels
# There should probably be a user_join_* that takes one user, multiple channels
# However, it seems that so far, Unreal splits both PART and JOIN (non-SJOIN)
# into multiple events/cmds. The reason is unclear.
# Other ircds may not do so. 
# There is also KICK. some IRCds allow KICK #chan user1,user2,...
# Unreal it's _supposed_ to work, but it does not.

	my ($user, $chanlist, $reason) = @_;
	my @chans;
	foreach my $cn (@$chanlist) {
		push @chans, { CHAN => $cn };
	
	}

	my $uid = get_user_id($user);
	my $seq = $ircline;

	$get_joinpart_lock->execute; $get_joinpart_lock->finish;

	foreach my $chan (@chans) {
		my $cn = $chan->{CHAN};
		$joinpart2->execute($seq, 0, 0, $uid, $cn, $seq, $seq+1000);
		$get_user_count->execute($cn);
		$chan->{COUNT} = $get_user_count->fetchrow_array;
	}

	$unlock_tables->execute; $unlock_tables->finish;
	
	foreach my $chan (@chans) {
		channel_emptied($chan) if $chan->{COUNT} == 0;
	}
}

sub channel_emptied($) {
	my ($chan) = @_;

	botserv::bot_part_if_needed(undef, $chan, 'Nobody\'s here', 1);
	$chan_delete->execute($chan->{CHAN});
	$wipe_bans->execute($chan->{CHAN});
}

sub process_kick($$$$) {
	my ($src, $cn, $target, $reason) = @_;
	user_part($target, $cn, 'Kicked by '.$src.' ('.$reason.')');
=cut
	net::debug("detected a KICK $src $cn $target $reason");
	my $chan = { CHAN => $cn };
	if (!is_agent($src)) and ({modes::splitmodes(get_modelock($chan))}->{Q}->[0] eq '+')) {
		net::kill(agent($chan), $src, "War script detected (kicked $target past +Q)");
		net::invite(agent($chan), $cn, $target);
		net::svsjoin(undef, $target, $cn);
		unban_user($chan, { NICK => $target });
	}
=cut
}

sub chan_mode($$$$) {
	my ($src, $cn, $modes, $args) = @_;
	my $user = { NICK => $src };
	my $chan = { CHAN => $cn };
	my ($sign, $num);
	
	# XXX This is not quite right, but maybe it's good enough.
	my $mysync = ($src =~ /\./ ? 0 : 1);
	
	if($modes !~ /^[bevhoaq+-]+$/ and (!synced() or $mysync)) {
		do_modelock($chan, "$modes $args");
	}
	
	my $opguard = (!current_message->{SYNC} and chk_flag($chan, F_OPGUARD, 1));
	
	my @perms = ('VOICE', 'HALFOP', 'OP', 'PROTECT');
	my $unmodes = '-';
	my @unargs;
	
	my @modes = split(//, $modes);
	my @args = split(/ /, $args);

	foreach my $mode (@modes) {
		if($mode eq '+') { $sign = 1; next; }
		if($mode eq '-') { $sign = 0; next; }
		
		my $arg = shift(@args) if($mode =~ /^[bekfjLlvhoaq]$/);
		my $auser = { NICK => $arg };
		
		if($mode =~ /^[vhoaq]$/) {
			next if $arg eq '';
			next if is_agent($arg);
			$num = 0 if $mode eq 'v';
			$num = 1 if $mode eq 'h';
			$num = 2 if $mode eq 'o';
			$num = 3 if $mode eq 'a';
			$num = 4 if $mode eq 'q';
		
			if($opguard and $sign == 1 and
				!can_keep_op($user, $chan, $auser, $mode)
			) {
				$unmodes .= $mode;
				push @unargs, $arg;
			} else {
				my $nid = get_user_id($auser) or return;
				my ($r, $i);
				do {
					if($sign) {
						$r = $chop->execute((2**$num), (2**$num), $nid, $cn);
					} else {
						$r = $chdeop->execute((2**$num), (2**$num), $nid, $cn);
					}
					$i++;
				} while($r==0 and $i<10);
			}
		}
		if ($mode eq 'b') {
			next if $arg eq '';
			process_ban($cn, $arg, $src, 0, $sign);
		}
		if ($mode eq 'e') {
			next if $arg eq '';
			process_ban($cn, $arg, $src, 128, $sign);
		}
		if ($mode eq 'I') {
			next;# if $arg eq '';
			#process_ban($cn, $arg, $src, 128, $sign);
		}
	}
	net::setmode(agent($chan), $cn, $unmodes, join(' ', @unargs)) if($opguard and @unargs);
}

sub process_ban($$$$) {
	my ($cn, $arg, $src, $type, $sign) = @_;
	
	$arg =~ tr/\*\?/\%\_/;
	
	if ($sign > 0) {
		$add_ban->execute($cn, $arg, $src, $type);
	} else {
		$delete_ban->execute($cn, $arg, $type);
	}
}

sub chan_topic {
	my ($src, $cn, $setter, $time, $topic) = @_;
	my $chan = { CHAN => $cn };
	my $suser = { NICK => $setter, AGENT => agent($chan) };

	return if chk_flag($chan, F_CLOSE, 1);
	
	if(!synced()) {
		$get_topic->execute($cn);
		my ($ntopic, $nsetter, $ntime) = $get_topic->fetchrow_array;
		if($topic ne '' and $time == $ntime or can_do($chan, 'SETTOPIC', 0)) {
			$set_topic->execute($topic, $setter, $time, $cn);
		} else {
			net::settopic(agent($chan), $cn, $nsetter, $ntime, $ntopic);
		}
	}
	
	elsif(lc($src) ne lc($setter) or can_do($chan, 'SETTOPIC', undef, $suser)) {
		$set_topic->execute($topic, $setter, $time, $cn);
	} else {
		$get_topic->execute($cn);
		my ($ntopic, $nsetter, $ntime) = $get_topic->fetchrow_array;
		net::settopic(agent($chan), $cn, $nsetter, $ntime, $ntopic);
	}
}

sub eos(;$) {
	my ($server) = @_;
	my $gsa;
	
	$get_all_closed_chans->execute(F_DRONE|F_CLOSE);
	while(my ($cn, $type, $reason, $opnick, $time) = $get_all_closed_chans->fetchrow_array) {
		my $chan = { CHAN => $cn };
		
		my $cmsg = " is closed [$opnick ".misc::gmtime2($time)."]: $reason";
		if($type == F_DRONE) {
			chan_kill($chan, $cn.$cmsg);
		} else {
			net::settopic(agent($chan), $cn, $opnick, $time, "Channel".$cmsg);
			clear_users($chan, "Channel".$cmsg);
		}
	}
	
	while($chanuser_table > 0) { }
	
	$get_eos_lock->execute(); $get_eos_lock->finish;
	$get_akick_all->execute();
	if($server) {
		$get_status_all_server->execute($server);
		$gsa = $get_status_all_server;
	} else {
		$get_status_all->execute();
		$gsa = $get_status_all;
	}
	#$unlock_tables->execute(); $unlock_tables->finish;

	while(my @akick = $get_akick_all->fetchrow_array) {
		akickban(@akick);
	}

	$get_modelock_all->execute();
	while(my ($cn, $modes, $ml) = $get_modelock_all->fetchrow_array) {
		$ml = do_modelock_fast($cn, '', $modes, $ml);
		net::setmode(agent({CHAN=>$cn}), $cn, $ml) if $ml;
	}

	while(my ($cn, $cflags, $agent, $nick, $uid, $uflags, $level, $op, $neverop) = $gsa->fetchrow_array) {
		my $user = { NICK => $nick, ID => $uid };
		#next if nickserv::chk_user_flag($user, nickserv::UF_FINISHED);
		$agent = $csnick unless $agent;
		my $chan = { CHAN => $cn, FLAGS => $cflags, AGENT => $agent };
		
		set_modes($user, $chan, $level, ($cflags & F_OPGUARD)) if not $neverop and $ops[$level] != $op and not $cflags & (F_FREEZE | F_CLOSE | F_DRONE);
		do_welcome($user, $chan);
	}

	$nickserv::set_user_flag_all->execute(nickserv::UF_FINISHED());
	$unlock_tables->execute(); $unlock_tables->finish;
}

1;
