#	This file is part of SurrealServices.
#
#	SurrealServices is free software; you can redistribute it and/or modify
#	it under the terms of the GNU General Public License as published by
#	the Free Software Foundation; either version 2 of the License, or
#	(at your option) any later version.
#
#	SurrealServices is distributed in the hope that it will be useful,
#	but WITHOUT ANY WARRANTY; without even the implied warranty of
#	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#	GNU General Public License for more details.
#
#	You should have received a copy of the GNU General Public License
#	along with SurrealServices; if not, write to the Free Software
#	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
package memoserv;

use strict;
use DBI qw(:sql_types);
#use constant {
#	READ => 1,
#	DEL => 2,
#	ACK => 4,
#	NOEXP => 8
#};

use SrSv::Agent qw(is_agent);

*get_user_id = \&nickserv::get_user_id;
*get_user_nick = \&nickserv::get_user_nick;
*is_identified = \&nickserv::is_identified;
*notice = \&nickserv::notice;

use constant (
	MAX_MEMO_LEN => 400
);

our $msnick = 'MemoServ';

my $err_deny = 'Permission denied.';

our (
	$send_memo, $send_chan_memo, $get_chan_recipients,

	$get_memo_list,

	$get_memo, $get_memo_full, $get_memo_count, $get_unread_memo_count,

	$set_flag,

	$delete_memo, $purge_memos, $delete_all_memos,
	$memo_chgroot,

	$add_ignore, $del_ignore, $list_ignore, $chk_ignore, $count_ignore, $wipe_ignore,
	$drop_ignore, $ignore_chgroot1, $ignore_chgroot2, $adjust_ignore, $consolidate_ignore
);

sub init() {
	$send_memo = $services::dbh->prepare("INSERT INTO memo SELECT ?, id, NULL, UNIX_TIMESTAMP(), NULL, ? FROM nickreg WHERE nick=?");
	$send_chan_memo = $services::dbh->prepare("INSERT INTO memo SELECT ?, nickreg.id, ?, ?, NULL, ? FROM chanacc, nickreg
		WHERE chanacc.chan=? AND chanacc.level >= ? AND chanacc.nrid=nickreg.id
		AND !(nickreg.flags & ". nickserv::F_NOMEMO() . ")");
	$get_chan_recipients = $services::dbh->prepare("SELECT user.nick FROM user, nickid, nickreg, chanacc WHERE
		user.id=nickid.id AND nickid.nrid=chanacc.nrid AND chanacc.nrid=nickreg.id AND chanacc.chan=?
		AND level >= ? AND
		!(nickreg.flags & ". nickserv::F_NOMEMO() . ")");

	$get_memo_list = $services::dbh->prepare("SELECT memo.src, memo.chan, memo.time, memo.flag, memo.msg FROM memo, nickreg WHERE nickreg.nick=? AND memo.dstid=nickreg.id ORDER BY memo.time ASC");

	$get_memo = $services::dbh->prepare("SELECT memo.src, memo.chan, memo.time FROM memo, nickreg WHERE nickreg.nick=? AND memo.dstid=nickreg.id ORDER BY memo.time ASC LIMIT 1 OFFSET ?");
	$get_memo->bind_param(2, 0, SQL_INTEGER);
	$get_memo_full = $services::dbh->prepare("SELECT memo.src, memo.chan, memo.time, memo.flag, memo.msg FROM memo, nickreg WHERE nickreg.nick=? AND memo.dstid=nickreg.id ORDER BY memo.time ASC LIMIT 1 OFFSET ?");
	$get_memo_full->bind_param(2, 0, SQL_INTEGER);
	$get_memo_count = $services::dbh->prepare("SELECT COUNT(*) FROM memo, nickreg WHERE nickreg.nick=? AND memo.dstid=nickreg.id");
	$get_unread_memo_count = $services::dbh->prepare("SELECT COUNT(*) FROM memo, nickreg WHERE nickreg.nick=? AND memo.dstid=nickreg.id AND memo.flag=0");

	$set_flag = $services::dbh->prepare("UPDATE memo, nickreg SET memo.flag=? WHERE memo.src=? AND nickreg.nick=? AND memo.dstid=nickreg.id AND memo.chan=? AND memo.time=?");

	$delete_memo = $services::dbh->prepare("DELETE FROM memo USING memo, nickreg WHERE memo.src=? AND nickreg.nick=? AND memo.dstid=nickreg.id AND memo.chan=? AND memo.time=?");
	$purge_memos = $services::dbh->prepare("DELETE FROM memo USING memo, nickreg WHERE nickreg.nick=? AND memo.dstid=nickreg.id AND memo.flag=1");
	$delete_all_memos = $services::dbh->prepare("DELETE FROM memo USING memo, nickreg WHERE nickreg.nick=? AND memo.dstid=nickreg.id");

	$add_ignore = $services::dbh->prepare("INSERT INTO ms_ignore SELECT nickreg.id, ignorenick.id, UNIX_TIMESTAMP(), ? FROM nickreg, nickreg AS ignorenick WHERE nickreg.nick=? AND ignorenick.nick=?");
	$del_ignore = $services::dbh->prepare("DELETE FROM ms_ignore USING ms_ignore, nickreg, nickreg AS ignorenick WHERE nickreg.nick=? AND ms_ignore.nrid=nickreg.id AND ms_ignore.id=? OR ignorenick.nick=? AND ms_ignore.ignoreid=ignorenick.id");
	$list_ignore = $services::dbh->prepare("SELECT ignorenick.nick, ms_ignore.time, ms_ignore.id FROM ms_ignore, nickreg, nickreg AS ignorenick WHERE nickreg.nick=? AND ms_ignore.nrid=nickreg.id AND ms_ignore.ignoreid=ignorenick.id ORDER BY ms_ignore.id");
	$chk_ignore = $services::dbh->prepare("SELECT 1 FROM ms_ignore, nickreg, nickreg AS ignorenick WHERE nickreg.nick=? AND ms_ignore.nrid=nickreg.id AND ignorenick.nick=? AND ms_ignore.ignoreid=ignorenick.id");
	$count_ignore = $services::dbh->prepare("SELECT COUNT(*) FROM ms_ignore, nickreg WHERE nickreg.nick=? AND ms_ignore.nrid=nickreg.id");
	$wipe_ignore = $services::dbh->prepare("DELETE FROM ms_ignore USING ms_ignore, nickreg WHERE nickreg.nick=? AND ms_ignore.nrid=nickreg.id");

	$adjust_ignore = $services::dbh->prepare("UPDATE ms_ignore, nickreg SET ms_ignore.id=ms_ignore.id+? WHERE nickreg.nick=? AND ms_ignore.nrid=nickreg.id");
	$consolidate_ignore = $services::dbh->prepare("UPDATE ms_ignore, nickreg SET ms_ignore.id=ms_ignore.id-1 WHERE nickreg.nick=? AND ms_ignore.nrid=nickreg.id and ms_ignore.id>?");
}

### MEMOSERV COMMANDS ###

sub dispatch($$$) {
	my ($src, $dst, $msg) = @_;
	$msg =~ s/^\s+//;
	my @args = split(/\s+/, $msg);
	my $cmd = shift @args;

	my $user = { NICK => $src, AGENT => $msnick };

	return if operserv::flood_check($user);

	if($cmd =~ /^send$/i) {
		if(@args >= 2) {
			my @args = split(/\s+/, $msg, 3);
			ms_send($user, $args[1], $args[2], 0);
		} else {
			notice($user, 'Syntax: SEND <recipient> <message>');
		}
	}
	elsif($cmd =~ /^csend$/i) {
		if(@args >= 3 and $args[1] =~ /^(?:[uvhas]op|co?f(ounder)?|founder)$/i) {
			my @args = split(/\s+/, $msg, 4);
			my $level = chanserv::xop_byname($args[2]);
			ms_send($user, $args[1], $args[3], $level);
		} else {
			notice($user, 'Syntax: CSEND <recipient> <uop|vop|hop|aop|sop|cf|founder> <message>');
		}
	}
	elsif($cmd =~ /^read$/i) {
		if(@args == 1 and (lc($args[0]) eq 'last' or $args[0] > 0)) {
			ms_read($user, $args[0]);
		} else {
			notice($user, 'Syntax: READ <num|LAST>');
		}
	}
	elsif($cmd =~ /^list$/i) {
		ms_list($user);
	}
	elsif($cmd =~ /^del(ete)?$/i) {
		if(@args == 1 and (lc($args[0]) eq 'all' or $args[0] > 0)) {
			ms_delete($user, $args[0]);
		} else {
			notice($user, 'Syntax: DELETE <num|ALL>');
		}
	}
	elsif($cmd =~ /^ign(ore)?$/i) {
		my $cmd2 = shift @args;
		if($cmd2 =~ /^a(dd)?$/i) {
			if(@args == 1) {
				ms_ignore_add($user, $args[0]);
			}
			else {
				notice($user, 'Syntax: IGNORE ADD <nick>');
			}
		}
		elsif($cmd2 =~ /^d(el)?$/i) {
			if(@args == 1) {
				ms_ignore_del($user, $args[0]);
			}
			else {
				notice($user, 'Syntax: IGNORE DEL [nick|num]');
			}
		}
		elsif($cmd2 =~ /^l(ist)?$/i) {
			ms_ignore_list($user);
		}
		else {
			notice($user, 'Syntax: IGNORE <ADD|DEL|LIST> [nick|num]');
		}
	}
	elsif($cmd =~ /^help$/i) {
		help::sendhelp($msnick, $src, lc $msnick, @args);
	}
	else {
		notice($user, "Unrecognized command.  For help, type: \002/ms help\002");
	}
}

sub ms_send($$$$) {
	my ($user, $dst, $msg, $level) = @_;
	my $src = get_user_nick($user);

	my $root = auth($user) or return;
	
	if(length($msg) > MAX_MEMO_LEN()) {
		notice($user, 'Memo too long. Maximum memo length is '.MAX_MEMO_LEN().' characters.');
		return;
	}

	if($dst =~ /^#/) {
		my $chan = { CHAN => $dst };
		unless(chanserv::is_registered($chan)) {
			notice($user, "$dst is not registered");
			return;
		}
		
		chanserv::can_do($chan, 'MEMO', undef, $user) or return;

		send_chan_memo($user, $chan, $msg, $level);
	} else {
		nickserv::chk_registered($user, $dst) or return;
		
		if (nickserv::chk_flag($dst, nickserv::F_NOMEMO(), +1)) {
			notice($user, "\002$dst\002 is not accepting memos.");
			return;
		}
		$chk_ignore->execute(nickserv::get_root_nick($dst), $root);
		if ($chk_ignore->fetchrow_array) {
			notice($user, "\002$dst\002 is not accepting memos.");
			return;
		}
			
		send_memo($src, $dst, $msg);
	}

	notice($user, "Your memo has been sent.");
}

sub ms_read($$) {
	my ($user, $num) = @_;
	my ($from, $chan, $time, $flag, $msg);
	my $src = get_user_nick($user);

	my $root = auth($user) or return;

	if(lc($num) eq 'last') {
		$get_memo_count->execute($root);
		($num) = $get_memo_count->fetchrow_array;
	}
	
	$get_memo_full->execute($root, $num-1);
	unless(($from, $chan, $time, $flag, $msg) = $get_memo_full->fetchrow_array) {
		notice($user, "That memo does not exist.");
		return;
	}
	$set_flag->execute(1, $from, $root, $chan, $time);

	notice($user, "Memo \002$num\002 from \002$from\002 ".
		($chan ? "to \002$chan\002 " : "to \002$root\002 ").
		"at ".misc::gmtime2($time), ' ', '  '.$msg, ' --');
}

sub ms_list($) {
	my ($user) = @_;
	my ($i, @data, $mnlen, $mclen);
	my $src = get_user_nick($user);

	my $root = auth($user) or return;

	my @reply = ("Memo list for \002$root\002.  To read, type \002/ms read <num>\002", ' ');

	$get_memo_list->execute($root);
	while(my ($from, $chan, $time, $flag, $msg) = $get_memo_list->fetchrow_array) {
		$i++;
		
		push @data, [
			($flag ? '' : "\002") . $i,
			$from, $chan, scalar(misc::gmtime2($time)),
			(length($msg) > 20 ? substr($msg, 0, 17) . '...' : $msg)
		];
	}

	unless(@data) {
		notice($user, "You have no memos.");
		return;
	}

	push @reply, fmt::columnar(@data);
	notice($user, @reply);
}

sub ms_delete($$) {
	my ($user, $num) = @_;
	my $src = get_user_nick($user);

	my $root = auth($user) or return;

	if(lc($num) eq 'all') {
		$delete_all_memos->execute($root);
		notice($user, 'All of your memos have been deleted.');
	} else {
		my ($from, $chan, $time);
		$get_memo->execute($root, $num-1);
		unless(($from, $chan, $time) = $get_memo->fetchrow_array) {
	                notice($user, "That memo does not exist.");
	                return;
	        }
		$delete_memo->execute($from, $root, $chan, $time);
		notice($user, "Memo $num has been deleted.");
	}
}		

sub ms_ignore_add($$) {
	my ($user, $nick) = @_;
	my $src = get_user_nick($user);

	unless(is_identified($user, $src) or adminserv::can_do($user, 'SERVOP')) {
		notice($user, $err_deny);
		return;
	}

	my $nickroot = nickserv::get_root_nick($nick);
	unless ($nickroot) {
		notice($user, "$nick is not registered");
		return;
	}

	my $srcroot = nickserv::get_root_nick($src);

	$count_ignore->execute($srcroot);
	my $count = $count_ignore->fetchrow_array;
	$count_ignore->finish();

	$add_ignore->execute(++$count, $srcroot, $nickroot);

	notice($user, "\002$nick\002 (\002$nickroot\002) added to \002$src\002 (\002$srcroot\002) memo ignore list.");
}

sub ms_ignore_del($$) {
	my ($user, $entry) = @_;
	my $src = get_user_nick($user);
	
	unless(is_identified($user, $src) or adminserv::can_do($user, 'SERVOP')) {
		notice($user, $err_deny);
		return;
	}
	my $srcroot = nickserv::get_root_nick($src);

	if (misc::isint($entry)) {
		my $ret = $del_ignore->execute($srcroot, $entry, undef);
		if($ret == 1) {
			notice($user, "Delete succeeded for ($srcroot): #$entry");
		}
		else {
			notice($user, "Delete failed for ($srcroot): #$entry. entry does not exist?");
		}
	}
	else {
		my $ret = $del_ignore->execute($srcroot, undef, $entry);
		if($ret == 1) {
			notice($user, "Delete succeeded for ($srcroot): $entry");
		}
		else {
			notice($user, "Delete failed for ($srcroot): $entry. entry does not exist?");
		}
	}
}

sub ms_ignore_list($) {
	my ($user) = @_;
	my $src = get_user_nick($user);
	
	unless(is_identified($user, $src) or adminserv::can_do($user, 'SERVOP')) {
		notice($user, $err_deny);
		return;
	}
	my $srcroot = nickserv::get_root_nick($src);

	my (@reply, @data);
	$list_ignore->execute($srcroot);
	while (my ($nick, $time, $id) = $list_ignore->fetchrow_array) {
		push @data, ["$id\)", $nick, "time added: ".misc::gmtime2($time)];
	}
	push @reply, "Memo ignore list for \002$src\002:", fmt::columnar(@data);
	net::notice($msnick, $src, @reply);
}

sub notify($;$) {
	my ($user, $root) = @_;
	my (@nicks);

	unless(ref($user)) {
		$user = { NICK => $user };
	}

	if($root) { @nicks = ($root) }
	else { @nicks = nickserv::get_id_nicks($user) }

	my $hasmemos;
	foreach my $n (@nicks) {
		$get_unread_memo_count->execute($n);
		my ($c) = $get_unread_memo_count->fetchrow_array;
		next unless $c;
		notice($user, "You have \002$c\002 unread memo(s). " . (@nicks > 1 ? "(\002$n\002) " : ''));
		$hasmemos = 1;
	}

	notice($user, "To view them, type: \002/ms list\002") if $hasmemos;
}

### DATABASE UTILITY FUNCTIONS ###

sub send_memo($$$) {
	my ($src, $dst, $msg) = @_;

	# This construct is intended to allow agents to send memos.
	# Unfortunately this is raceable against %nickserv::enforcers.
	# I don't want to change the %nickserv::enforcers decl tho, s/my/our/
	$src = (is_agent($src) ? $src : nickserv::get_root_nick($src));
	$dst = nickserv::get_root_nick($dst);

	$send_memo->execute($src, $msg, $dst);
	notice_all_nicks($dst, "You have a new memo from \002$src\002.  To read it, type: \002/ms read last\002");
}

sub send_chan_memo($$$$) {
	my ($user, $chan, $msg, $level) = @_;
	my $cn = $chan->{CHAN};
	my $src = get_user_nick($user);

	$send_chan_memo->execute($src, $cn, time(), $msg, $cn, $level);
	# "INSERT INTO memo SELECT ?, nick, ?, ?, 0, ? FROM chanacc WHERE chan=? AND level >= ?"
	
	$get_chan_recipients->execute($cn, $level);
	while(my ($u) = $get_chan_recipients->fetchrow_array) {
		net::notice($msnick, $u, "You have a new memo from \002$src\002 to \002$cn\002.  To read it, type: \002/ms read last\002");
	}
}

sub notice_all_nicks($$) {
	my ($nick, $msg) = @_;

	$nickserv::get_using_nicks->execute($nick);
	while(my ($u) = $nickserv::get_using_nicks->fetchrow_array) {
		net::notice($msnick, $u, $msg);
	}
}

sub auth($) {
	my ($user) = @_;
	my $src = get_user_nick($user);
	
	my $root = nickserv::get_root_nick($src);
        unless($root) {
                notice($user, "Your nick is not registered.");
                return 0;
        }

        unless(is_identified($user, $root)) {
                notice($user, $err_deny);
                return 0;
        }

	return $root;
}

### IRC EVENTS ###

1;
