#!/usr/bin/perl

#	This file is part of SurrealServices.
#
#	SurrealServices is free software; you can redistribute it and/or modify
#	it under the terms of the GNU General Public License as published by
#	the Free Software Foundation; either version 2 of the License, or
#	(at your option) any later version.
#
#	SurrealServices is distributed in the hope that it will be useful,
#	but WITHOUT ANY WARRANTY; without even the implied warranty of
#	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#	GNU General Public License for more details.
#
#	You should have received a copy of the GNU General Public License
#	along with SurrealServices; if not, write to the Free Software
#	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#  SurrealChat.net does not provide the Country/Allocation data,
#  is in no way associated with ludost.net,
#  nor are we providing a license to download/use it.
#  Be sure to direct availability/accuracy/licensing questions to ludost.net
use strict;
use DBI;
use Cwd 'abs_path';
use File::Basename;
use Date::Parse;

use lib '..';
use libs::config;

my $prefix = dirname(dirname(abs_path($0)).'../');

my $countrydb_url= 'http://ip.ludost.net/raw/country.db.gz';
my $srcname = 'country.db.gz';
my $unpackname = 'country.db';


# This MAY be implementable with an open of a pipe
# pipe the output of wget through gzip -d
# and then into the load-loop.
# It's a bit heavy to run directly from inside services however.
# I'd recommend it be run as a crontab script separate from services.

my (@stats, $date, $size);
if(@stats = stat("$prefix/data/$srcname")) {
	print "Checking for updated country data...\n";
	my $header = qx{wget --spider -S $countrydb_url 2>&1};
	($date) = ($header =~ /Last-Modified: (.*)/);
	($size) = ($header =~ /Content-Length: (.*)/);
}

if(@stats and $stats[7] == $size and $stats[9] >= str2time($date)) {
	print "Country data is up to date.\n";
} else {
	print "Downloading country data...\n";

	unlink "$prefix/data/$srcname";
	system("wget $countrydb_url -O $prefix/data/$srcname");
	unless(-e "$prefix/data/$srcname") {
		print "FATAL: Download failed.\n";
		exit;
	}
}

print "Decompressing...\n";
unlink "$prefix/data/$unpackname";
system("gunzip -c $prefix/data/$srcname >$prefix/data/$unpackname");
unless(open COUNTRYTABLE, "$prefix/data/$unpackname") {
	print "FATAL: Decompression failed.\n";
	exit;
}

my %conf = config::readHash("$prefix/config/services.conf");

print "Connecting to database...\n";

my $dbh;
eval { 
	$dbh = DBI->connect("DBI:mysql:".$conf{'mysql-db'}, $conf{'mysql-user'}, $conf{'mysql-pass'}, 
		{  AutoCommit => 1, RaiseError => 1 })
};

if($@) {
	print "FATAL: Can't connect to database:\n$@\n";
	print "You must edit config/services.conf and create a corresponding\nMySQL user and database!\n\n";
	exit;
}

print "Creating new table...\n";

$dbh->{RaiseError} = 1;
$dbh->{PrintError} = 1;

my $add_entry = $dbh->prepare("INSERT INTO newcountry SET low=INET_ATON(?), high=INET_ATON(?), country=?");

$dbh->do("DROP TABLE IF EXISTS newcountry");
$dbh->do(
"CREATE TABLE `newcountry` (
  `low` int unsigned NOT NULL default 0,
  `high` int unsigned NOT NULL default 0,
  `country` char(2) NOT NULL default '-',
  PRIMARY KEY (`low`, `high`)
) TYPE=MyISAM"
);

print "Inserting data...     ";

$| = 1;
my ($lines) = qx{wc -l $prefix/data/$unpackname};
my $div = int($lines/100);
my $i;

$dbh->do("ALTER TABLE `newcountry` DISABLE KEYS");
$dbh->do("LOCK TABLES newcountry WRITE");
while(my $x = <COUNTRYTABLE>) {
	if($i == 0 or !($i % $div)) {
		printf("\b\b\b\b%3d%", ($i/$lines)*100);
	}
	
	chomp $x;
	my ($low, $high, $country) = split(/ /, $x);
	$add_entry->execute($low, $high, $country);

	$i++;
}
$dbh->do("UNLOCK TABLES");
$dbh->do("ALTER TABLE `newcountry` ENABLE KEYS");
close COUNTRYTABLE;

print "\b\b\b\bdone.\nRemoving old table...\n";
$dbh->do("DROP TABLE IF EXISTS `oldcountry`");
print "Renaming new table...\n";
$dbh->{RaiseError} = 0;
$dbh->do("RENAME TABLE `country` TO `oldcountry`");
$dbh->do("RENAME TABLE `newcountry` TO `country`");
$dbh->do("DROP TABLE `oldcountry`");

print "Country table update complete.\n";

unlink "$prefix/data/$unpackname";

exit;
